/*
sv_pmove.c - server-side player physic
Copyright (C) 2010 Uncle Mike

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.
*/

#include "common.h"
#include "server.h"
#include "const.h"
#include "pm_local.h"

static qboolean has_update = false;

void SV_ClearPhysEnts( void )
{
	svgame.pmove->numtouch = 0;
	svgame.pmove->numvisent = 0;
	svgame.pmove->nummoveent = 0;
	svgame.pmove->numphysent = 0;
}

void SV_CopyPmtraceToGlobal( pmtrace_t *trace )
{
	svgame.globals->trace_allsolid = trace->allsolid;
	svgame.globals->trace_startsolid = trace->startsolid;
	svgame.globals->trace_fraction = trace->fraction;
	svgame.globals->trace_plane_dist = trace->plane.dist;
	svgame.globals->trace_flags = 0;
	svgame.globals->trace_inopen = trace->inopen;
	svgame.globals->trace_inwater = trace->inwater;
	VectorCopy( trace->endpos, svgame.globals->trace_endpos );
	VectorCopy( trace->plane.normal, svgame.globals->trace_plane_normal );
	svgame.globals->trace_hitgroup = trace->hitgroup;
	if( trace->ent == -1 ) svgame.globals->trace_ent = svgame.edicts;
	else svgame.globals->trace_ent = EDICT_NUM( svgame.pmove->physents[trace->ent].info );
}

qboolean SV_CopyEdictToPhysEnt( physent_t *pe, edict_t *ed )
{
	model_t	*mod = Mod_Handle( ed->v.modelindex );

	if( !mod ) return false;
	pe->player = false;

	if( ed->v.flags & ( FL_CLIENT|FL_FAKECLIENT ))
	{
		// client or bot
		Q_strncpy( pe->name, "player", sizeof( pe->name ));
		pe->player = NUM_FOR_EDICT( ed );
	}
	else
	{
		// otherwise copy the modelname
		Q_strncpy( pe->name, mod->name, sizeof( pe->name ));
	}

	if( mod->type == mod_studio )
	{
		pe->studiomodel = mod;
		pe->model = NULL;
	}
	else
	{
		pe->studiomodel = NULL;
		pe->model = mod;
	}

	pe->info = NUM_FOR_EDICT( ed );
	VectorCopy( ed->v.origin, pe->origin );
	VectorCopy( ed->v.angles, pe->angles );
	VectorCopy( ed->v.mins, pe->mins );
	VectorCopy( ed->v.maxs, pe->maxs );

	pe->solid = ed->v.solid;
	pe->rendermode = ed->v.rendermode;
	pe->skin = ed->v.skin;
	pe->frame = ed->v.frame;
	pe->sequence = ed->v.sequence;

	Q_memcpy( &pe->controller[0], &ed->v.controller[0], 4 * sizeof( byte ));
	Q_memcpy( &pe->blending[0], &ed->v.blending[0], 2 * sizeof( byte ));

	pe->movetype = ed->v.movetype;
	pe->takedamage = ed->v.takedamage;
	pe->team = ed->v.team;
	pe->classnumber = ed->v.playerclass;
	pe->blooddecal = 0;	// unused in GoldSrc

	// for mods
	pe->iuser1 = ed->v.iuser1;
	pe->iuser2 = ed->v.iuser2;
	pe->iuser3 = ed->v.iuser3;
	pe->iuser4 = ed->v.iuser4;
	pe->fuser1 = (svgame.movevars.studio_scale) ? ed->v.scale : ed->v.fuser1;
	pe->fuser2 = ed->v.fuser2;
	pe->fuser3 = ed->v.fuser3;
	pe->fuser4 = ed->v.fuser4;

	VectorCopy( ed->v.vuser1, pe->vuser1 );
	VectorCopy( ed->v.vuser2, pe->vuser2 );
	VectorCopy( ed->v.vuser3, pe->vuser3 );
	VectorCopy( ed->v.vuser4, pe->vuser4 );

	return true;
}

/*
====================
SV_AddLinksToPmove

collect solid entities
====================
*/
void SV_AddLinksToPmove( areanode_t *node, const vec3_t pmove_mins, const vec3_t pmove_maxs )
{
	link_t	*l, *next;
	edict_t	*check, *pl;
	physent_t	*pe;

	pl = EDICT_NUM( svgame.pmove->player_index + 1 );
	ASSERT( SV_IsValidEdict( pl ));

	// touch linked edicts
	for( l = node->solid_edicts.next; l != &node->solid_edicts; l = next )
	{
		next = l->next;
		check = EDICT_FROM_AREA( l );

		if( check->v.groupinfo != 0 )
		{
			if(( !svs.groupop && (check->v.groupinfo & pl->v.groupinfo ) == 0) ||
			( svs.groupop == 1 && ( check->v.groupinfo & pl->v.groupinfo ) != 0 ))
				continue;
		}

		if( check->v.owner == pl || check->v.solid == SOLID_TRIGGER )
			continue; // player or player's own missile

		if( svgame.pmove->numvisent < MAX_PHYSENTS )
		{
			pe = &svgame.pmove->visents[svgame.pmove->numvisent];
			if( SV_CopyEdictToPhysEnt( pe, check ))
				svgame.pmove->numvisent++;
		}

		if( check->v.solid == SOLID_NOT && ( check->v.skin == 0 || check->v.modelindex == 0 ))
			continue;

		// ignore monsterclip brushes
		if(( check->v.flags & FL_MONSTERCLIP ) && check->v.solid == SOLID_BSP )
			continue;

		if( check == pl ) continue;	// himself

		if((( check->v.flags & FL_CLIENT ) && check->v.health <= 0 ) || check->v.deadflag == DEAD_DEAD )
			continue;	// dead body

		if( VectorIsNull( check->v.size )) continue;

		if( !BoundsIntersect( pmove_mins, pmove_maxs, check->v.absmin, check->v.absmax ))
			continue;

		if( svgame.pmove->numphysent < MAX_PHYSENTS )
		{
			pe = &svgame.pmove->physents[svgame.pmove->numphysent];

			if( SV_CopyEdictToPhysEnt( pe, check ))
				svgame.pmove->numphysent++;
		}
	}
	
	// recurse down both sides
	if( node->axis == -1 ) return;

	if( pmove_maxs[node->axis] > node->dist )
		SV_AddLinksToPmove( node->children[0], pmove_mins, pmove_maxs );
	if( pmove_mins[node->axis] < node->dist )
		SV_AddLinksToPmove( node->children[1], pmove_mins, pmove_maxs );
}

/*
====================
SV_AddLaddersToPmove
====================
*/
void SV_AddLaddersToPmove( areanode_t *node, const vec3_t pmove_mins, const vec3_t pmove_maxs )
{
	link_t	*l, *next;
	edict_t	*check;
	physent_t	*pe;
	
	// get water edicts
	for( l = node->water_edicts.next; l != &node->water_edicts; l = next )
	{
		next = l->next;
		check = EDICT_FROM_AREA( l );

		if( check->v.solid != SOLID_NOT ) // disabled ?
			continue;

		// only brushes can have special contents
		if( Mod_GetType( check->v.modelindex ) != mod_brush )
			continue;

		if( !BoundsIntersect( pmove_mins, pmove_maxs, check->v.absmin, check->v.absmax ))
			continue;

		if( svgame.pmove->nummoveent == MAX_MOVEENTS )
			return;

		pe = &svgame.pmove->moveents[svgame.pmove->nummoveent];
		if( SV_CopyEdictToPhysEnt( pe, check ))
			svgame.pmove->nummoveent++;
	}
	
	// recurse down both sides
	if( node->axis == -1 ) return;
	
	if( pmove_maxs[node->axis] > node->dist )
		SV_AddLaddersToPmove( node->children[0], pmove_mins, pmove_maxs );
	if( pmove_mins[node->axis] < node->dist )
		SV_AddLaddersToPmove( node->children[1], pmove_mins, pmove_maxs );
}

static void pfnParticle( float *origin, int color, float life, int zpos, int zvel )
{
	int	v;

	if( !origin )
	{
		MsgDev( D_ERROR, "SV_StartParticle: NULL origin. Ignored\n" );
		return;
	}

	BF_WriteByte( &sv.reliable_datagram, svc_particle );
	BF_WriteBitVec3Coord( &sv.reliable_datagram, origin );
	BF_WriteChar( &sv.reliable_datagram, 0 ); // no x-vel
	BF_WriteChar( &sv.reliable_datagram, 0 ); // no y-vel
	v = bound( -128, (zpos * zvel) * 16, 127 );
	BF_WriteChar( &sv.reliable_datagram, v ); // write z-vel
	BF_WriteByte( &sv.reliable_datagram, 1 );
	BF_WriteByte( &sv.reliable_datagram, color );
	BF_WriteByte( &sv.reliable_datagram, bound( 0, life * 8, 255 ));
}

static int pfnTestPlayerPosition( float *pos, pmtrace_t *ptrace )
{
	pmtrace_t trace;

	trace = PM_PlayerTrace( svgame.pmove, pos, pos, PM_NORMAL, svgame.pmove->usehull, -1, NULL );
	if( ptrace ) *ptrace = trace; 

	return PM_TestPlayerPosition( svgame.pmove, pos, NULL );
}

static void pfnStuckTouch( int hitent, pmtrace_t *tr )
{
	physent_t	*pe;
	float	*mins, *maxs;
	int	i;

	ASSERT( hitent >= 0 && hitent < svgame.pmove->numphysent );
	pe = &svgame.pmove->physents[hitent];
	mins = svgame.pmove->player_mins[svgame.pmove->usehull];
	maxs = svgame.pmove->player_maxs[svgame.pmove->usehull];

	if( !PM_TraceModel( svgame.pmove, pe, svgame.pmove->origin, mins, maxs, svgame.pmove->origin, tr, 0 ))
		return;	// not stuck

	tr->ent = hitent;

	for( i = 0; i < svgame.pmove->numtouch; i++ )
	{
		if( svgame.pmove->touchindex[i].ent == tr->ent )
			break;
	}

	if( i != svgame.pmove->numtouch ) return;
	VectorCopy( svgame.pmove->velocity, tr->deltavelocity );

	if( svgame.pmove->numtouch >= MAX_PHYSENTS )
	{
		MsgDev( D_ERROR, "PM_StuckTouch: MAX_TOUCHENTS limit exceeded\n" );
		return;
	}
	svgame.pmove->touchindex[svgame.pmove->numtouch++] = *tr;
}

static int pfnPointContents( float *p, int *truecontents )
{
	int	cont, truecont;

	truecont = cont = SV_TruePointContents( p );
	if( truecontents ) *truecontents = truecont;

	if( cont <= CONTENTS_CURRENT_0 && cont >= CONTENTS_CURRENT_DOWN )
		cont = CONTENTS_WATER;
	return cont;
}

static int pfnTruePointContents( float *p )
{
	return SV_TruePointContents( p );
}

static int pfnHullPointContents( struct hull_s *hull, int num, float *p )
{
	return PM_HullPointContents( hull, num, p );
}

static pmtrace_t pfnPlayerTrace( float *start, float *end, int traceFlags, int ignore_pe )
{
	return PM_PlayerTrace( svgame.pmove, start, end, traceFlags, svgame.pmove->usehull, ignore_pe, NULL );
}

static pmtrace_t *pfnTraceLine( float *start, float *end, int flags, int usehull, int ignore_pe )
{
	static pmtrace_t	tr;

	tr = PM_PlayerTrace( svgame.pmove, start, end, flags, usehull, ignore_pe, NULL );
	return &tr;
}

static hull_t *pfnHullForBsp( physent_t *pe, float *offset )
{
	return PM_HullForBsp( pe, svgame.pmove, offset );
}

static float pfnTraceModel( physent_t *pEnt, float *start, float *end, trace_t *trace )
{
	pmtrace_t	pmtrace;

	PM_TraceModel( svgame.pmove, pEnt, start, vec3_origin, vec3_origin, end, &pmtrace, PM_STUDIO_BOX );

	// copy pmtrace_t to trace_t
	if( trace )
	{
		// NOTE: if pmtrace.h is changed is must be changed too
		Q_memcpy( trace, &pmtrace, sizeof( *trace ));
		trace->hitgroup = pmtrace.hitgroup;
		trace->ent = NULL;
	}

	return pmtrace.fraction;
}

static const char *pfnTraceTexture( int ground, float *vstart, float *vend )
{
	physent_t *pe;

	if( ground < 0 || ground >= svgame.pmove->numphysent )
		return NULL; // bad ground

	pe = &svgame.pmove->physents[ground];
	return PM_TraceTexture( pe, vstart, vend );
}			

static void pfnPlaySound( int channel, const char *sample, float volume, float attenuation, int fFlags, int pitch )
{
	edict_t	*ent;

	ent = EDICT_NUM( svgame.pmove->player_index + 1 );
	if( !SV_IsValidEdict( ent )) return;

	SV_StartSound( ent, channel, sample, volume, attenuation, fFlags, pitch );
}

static void pfnPlaybackEventFull( int flags, int clientindex, word eventindex, float delay, float *origin,
	float *angles, float fparam1, float fparam2, int iparam1, int iparam2, int bparam1, int bparam2 )
{
	edict_t	*ent;

	ent = EDICT_NUM( clientindex + 1 );
	if( !SV_IsValidEdict( ent )) return;

	SV_PlaybackEventFull( flags, ent, eventindex,
		delay, origin, angles,
		fparam1, fparam2,
		iparam1, iparam2,
		bparam1, bparam2 );
}

static pmtrace_t pfnPlayerTraceEx( float *start, float *end, int traceFlags, pfnIgnore pmFilter )
{
	return PM_PlayerTrace( svgame.pmove, start, end, traceFlags, svgame.pmove->usehull, -1, pmFilter );
}

static int pfnTestPlayerPositionEx( float *pos, pmtrace_t *ptrace, pfnIgnore pmFilter )
{
	pmtrace_t trace;

	trace = PM_PlayerTrace( svgame.pmove, pos, pos, PM_STUDIO_BOX, svgame.pmove->usehull, -1, pmFilter );
	if( ptrace ) *ptrace = trace; 

	return PM_TestPlayerPosition( svgame.pmove, pos, pmFilter );
}

static pmtrace_t *pfnTraceLineEx( float *start, float *end, int flags, int usehull, pfnIgnore pmFilter )
{
	static pmtrace_t	tr;

	tr = PM_PlayerTrace( svgame.pmove, start, end, flags, usehull, -1, pmFilter );
	return &tr;
}

/*
===============
SV_InitClientMove

===============
*/
void SV_InitClientMove( void )
{
	int	i;

	Pmove_Init ();

	svgame.pmove->server = true;
	svgame.pmove->movevars = &svgame.movevars;
	svgame.pmove->runfuncs = false;

	Mod_SetupHulls( svgame.player_mins, svgame.player_maxs );
	
	// enumerate client hulls
	for( i = 0; i < 4; i++ )
	{
		if( svgame.dllFuncs.pfnGetHullBounds( i, svgame.player_mins[i], svgame.player_maxs[i] ))
			MsgDev( D_NOTE, "SV: hull%i, player_mins: %g %g %g, player_maxs: %g %g %g\n", i,
			svgame.player_mins[i][0], svgame.player_mins[i][1], svgame.player_mins[i][2],
			svgame.player_maxs[i][0], svgame.player_maxs[i][1], svgame.player_maxs[i][2] );
	}

	Q_memcpy( svgame.pmove->player_mins, svgame.player_mins, sizeof( svgame.player_mins ));
	Q_memcpy( svgame.pmove->player_maxs, svgame.player_maxs, sizeof( svgame.player_maxs ));

	// common utilities
	svgame.pmove->PM_Info_ValueForKey = Info_ValueForKey;
	svgame.pmove->PM_Particle = pfnParticle;
	svgame.pmove->PM_TestPlayerPosition = pfnTestPlayerPosition;
	svgame.pmove->Con_NPrintf = Con_NPrintf;
	svgame.pmove->Con_DPrintf = Con_DPrintf;
	svgame.pmove->Con_Printf = Con_Printf;
	svgame.pmove->Sys_FloatTime = Sys_DoubleTime;
	svgame.pmove->PM_StuckTouch = pfnStuckTouch;
	svgame.pmove->PM_PointContents = pfnPointContents;
	svgame.pmove->PM_TruePointContents = pfnTruePointContents;
	svgame.pmove->PM_HullPointContents = pfnHullPointContents; 
	svgame.pmove->PM_PlayerTrace = pfnPlayerTrace;
	svgame.pmove->PM_TraceLine = pfnTraceLine;
	svgame.pmove->RandomLong = Com_RandomLong;
	svgame.pmove->RandomFloat = Com_RandomFloat;
	svgame.pmove->PM_GetModelType = pfnGetModelType;
	svgame.pmove->PM_GetModelBounds = pfnGetModelBounds;	
	svgame.pmove->PM_HullForBsp = pfnHullForBsp;
	svgame.pmove->PM_TraceModel = pfnTraceModel;
	svgame.pmove->COM_FileSize = COM_FileSize;
	svgame.pmove->COM_LoadFile = COM_LoadFile;
	svgame.pmove->COM_FreeFile = COM_FreeFile;
	svgame.pmove->memfgets = COM_MemFgets;
	svgame.pmove->PM_PlaySound = pfnPlaySound;
	svgame.pmove->PM_TraceTexture = pfnTraceTexture;
	svgame.pmove->PM_PlaybackEventFull = pfnPlaybackEventFull;
	svgame.pmove->PM_PlayerTraceEx = pfnPlayerTraceEx;
	svgame.pmove->PM_TestPlayerPositionEx = pfnTestPlayerPositionEx;
	svgame.pmove->PM_TraceLineEx = pfnTraceLineEx;

	// initalize pmove
	svgame.dllFuncs.pfnPM_Init( svgame.pmove );
}

static void PM_CheckMovingGround( edict_t *ent, float frametime )
{
	SV_UpdateBaseVelocity( ent );

	if(!( ent->v.flags & FL_BASEVELOCITY ))
	{
		// apply momentum (add in half of the previous frame of velocity first)
		VectorMA( ent->v.velocity, 1.0f + (frametime * 0.5f), ent->v.basevelocity, ent->v.velocity );
		VectorClear( ent->v.basevelocity );
	}
	ent->v.flags &= ~FL_BASEVELOCITY;
}

static void SV_SetupPMove( playermove_t *pmove, edict_t *clent, usercmd_t *ucmd, const char *physinfo )
{
	vec3_t	absmin, absmax;
	int	i;

	svgame.globals->frametime = (ucmd->msec * 0.001f);

	pmove->player_index = NUM_FOR_EDICT( clent ) - 1;
	pmove->multiplayer = (sv_maxclients->integer > 1) ? true : false;
	pmove->time = sv.time; // probably never used
	VectorCopy( clent->v.origin, pmove->origin );
	VectorCopy( clent->v.v_angle, pmove->angles );
	VectorCopy( clent->v.v_angle, pmove->oldangles );
	VectorCopy( clent->v.velocity, pmove->velocity );
	VectorCopy( clent->v.basevelocity, pmove->basevelocity );
	VectorCopy( clent->v.view_ofs, pmove->view_ofs );
	VectorClear( pmove->movedir );
	pmove->flDuckTime = clent->v.flDuckTime;
	pmove->bInDuck = clent->v.bInDuck;
	pmove->usehull = (clent->v.flags & FL_DUCKING) ? 1 : 0; // reset hull
	pmove->flTimeStepSound = clent->v.flTimeStepSound;
	pmove->iStepLeft = clent->v.iStepLeft;
	pmove->flFallVelocity = clent->v.flFallVelocity;
	pmove->flSwimTime = clent->v.flSwimTime;
	VectorCopy( clent->v.punchangle, pmove->punchangle );
	pmove->flSwimTime = clent->v.flSwimTime;
	pmove->flNextPrimaryAttack = 0.0f; // not used by PM_ code
	pmove->effects = clent->v.effects;
	pmove->flags = clent->v.flags;
	pmove->gravity = clent->v.gravity;
	pmove->friction = clent->v.friction;
	pmove->oldbuttons = clent->v.oldbuttons;
	pmove->waterjumptime = clent->v.teleport_time;
	pmove->dead = (clent->v.health <= 0.0f ) ? true : false;
	pmove->deadflag = clent->v.deadflag;
	pmove->spectator = 0; // spectator physic all execute on client
	pmove->movetype = clent->v.movetype;
	if( pmove->multiplayer ) pmove->onground = -1;
	pmove->waterlevel = clent->v.waterlevel;
	pmove->watertype = clent->v.watertype;
	pmove->maxspeed = svgame.movevars.maxspeed;
	pmove->clientmaxspeed = clent->v.maxspeed;
	pmove->iuser1 = clent->v.iuser1;
	pmove->iuser2 = clent->v.iuser2;
	pmove->iuser3 = clent->v.iuser3;
	pmove->iuser4 = clent->v.iuser4;
	pmove->fuser1 = clent->v.fuser1;
	pmove->fuser2 = clent->v.fuser2;
	pmove->fuser3 = clent->v.fuser3;
	pmove->fuser4 = clent->v.fuser4;
	VectorCopy( clent->v.vuser1, pmove->vuser1 );
	VectorCopy( clent->v.vuser2, pmove->vuser2 );
	VectorCopy( clent->v.vuser3, pmove->vuser3 );
	VectorCopy( clent->v.vuser4, pmove->vuser4 );
	pmove->cmd = *ucmd;	// setup current cmds	

	Q_strncpy( pmove->physinfo, physinfo, MAX_INFO_STRING );

	// setup physents
	pmove->numvisent = 0;
	pmove->numphysent = 0;
	pmove->nummoveent = 0;

	for( i = 0; i < 3; i++ )
	{
		absmin[i] = clent->v.origin[i] - 256;
		absmax[i] = clent->v.origin[i] + 256;
	}

	SV_CopyEdictToPhysEnt( &svgame.pmove->physents[0], &svgame.edicts[0] );
	svgame.pmove->visents[0] = svgame.pmove->physents[0];
	svgame.pmove->numphysent = 1;	// always have world
	svgame.pmove->numvisent = 1;

	SV_AddLinksToPmove( sv_areanodes, absmin, absmax );
	SV_AddLaddersToPmove( sv_areanodes, absmin, absmax );
}

static void SV_FinishPMove( playermove_t *pmove, edict_t *clent )
{
	clent->v.teleport_time = pmove->waterjumptime;
	VectorCopy( pmove->origin, clent->v.origin );
	VectorCopy( pmove->view_ofs, clent->v.view_ofs );
	VectorCopy( pmove->velocity, clent->v.velocity );
	VectorCopy( pmove->basevelocity, clent->v.basevelocity );
	VectorCopy( pmove->punchangle, clent->v.punchangle );
	clent->v.flFallVelocity = pmove->flFallVelocity;
	clent->v.oldbuttons = pmove->oldbuttons;
	clent->v.waterlevel = pmove->waterlevel;
	clent->v.watertype = pmove->watertype;
	clent->v.flTimeStepSound = pmove->flTimeStepSound;
	clent->v.flDuckTime = pmove->flDuckTime;
	clent->v.flSwimTime = pmove->flSwimTime;
	clent->v.iStepLeft = pmove->iStepLeft;
	clent->v.movetype = pmove->movetype;
	clent->v.friction = pmove->friction;
	clent->v.bInDuck = pmove->bInDuck;
	clent->v.gravity = pmove->gravity;
	clent->v.flags = pmove->flags;

	// copy back user variables
	clent->v.iuser1 = pmove->iuser1;
	clent->v.iuser2 = pmove->iuser2;
	clent->v.iuser3 = pmove->iuser3;
	clent->v.iuser4 = pmove->iuser4;
	clent->v.fuser1 = pmove->fuser1;
	clent->v.fuser2 = pmove->fuser2;
	clent->v.fuser3 = pmove->fuser3;
	clent->v.fuser4 = pmove->fuser4;
	VectorCopy( pmove->vuser1, clent->v.vuser1 );
	VectorCopy( pmove->vuser2, clent->v.vuser2 );
	VectorCopy( pmove->vuser3, clent->v.vuser3 );
	VectorCopy( pmove->vuser4, clent->v.vuser4 );

	if( svgame.pmove->onground == -1 )
		clent->v.flags &= ~FL_ONGROUND;
	else if( pmove->onground >= 0 && pmove->onground < pmove->numphysent )
	{
		clent->v.flags |= FL_ONGROUND;
		clent->v.groundentity = EDICT_NUM( svgame.pmove->physents[svgame.pmove->onground].info );
	}

	// angles
	// show 1/3 the pitch angle and all the roll angle	
	if( !clent->v.fixangle )
	{
		VectorCopy( pmove->angles, clent->v.v_angle );
		clent->v.angles[PITCH] = -( clent->v.v_angle[PITCH] / 3.0f );
		clent->v.angles[ROLL] = clent->v.v_angle[ROLL];
		clent->v.angles[YAW] = clent->v.v_angle[YAW];
	}
}

entity_state_t *SV_FindEntInPack( int index, client_frame_t *frame )
{
	entity_state_t	*state;
	int		i;	

	for( i = 0; i < frame->num_entities; i++ )
	{
		state = &svs.packet_entities[(frame->first_entity+i)%svs.num_client_entities];

		if( state->number == index )
			return state;
	}
	return NULL;
}

int SV_UnlagCheckTeleport( vec3_t old_pos, vec3_t new_pos )
{
	int	i;

	for( i = 0; i < 3; i++ )
	{
		if( fabs( old_pos[i] - new_pos[i] ) > 50 )
			return 1;
	}
	return 0;
}

void SV_SetupMoveInterpolant( sv_client_t *cl )
{
	int		i, j, clientnum;
	float		finalpush, lerp_msec;
	float		latency, temp, lerpFrac;
	client_frame_t	*frame, *frame2;
	entity_state_t	*state, *lerpstate;
	vec3_t		tempvec, tempvec2;
	sv_client_t	*check;
	sv_interp_t	*lerp;

	Q_memset( svgame.interp, 0, sizeof( svgame.interp ));
	has_update = false;

	// don't allow unlag in singleplayer
	if( sv_maxclients->integer <= 1 ) return;

	// unlag disabled by game request
	if( !svgame.dllFuncs.pfnAllowLagCompensation() || !sv_unlag->integer )
		return;

	// unlag disabled for current client
	if( cl->state != cs_spawned || !cl->lag_compensation )
		return;

	has_update = true;

	for( i = 0, check = svs.clients; i < sv_maxclients->integer; i++, cl++ )
	{
		if( check->state != cs_spawned || check == cl )
			continue;

		lerp = &svgame.interp[i];

		VectorCopy( check->edict->v.origin, lerp->oldpos );
		VectorCopy( check->edict->v.absmin, lerp->mins );
		VectorCopy( check->edict->v.absmax, lerp->maxs );
		lerp->active = true;
	}

	if( cl->latency > 1.5f )
		latency = 1.5f;
	else latency = cl->latency;

	temp = sv_maxunlag->value;

	if( temp > 0 && latency > temp )
		latency = temp;

	lerp_msec = cl->lastcmd.lerp_msec * 0.001f;
	if( lerp_msec > 0.1f ) lerp_msec = 0.1f;

	if( lerp_msec < cl->cl_updaterate )
		lerp_msec = cl->cl_updaterate;

	finalpush = sv_unlagpush->value + (( host.realtime - latency ) - lerp_msec );

	if( finalpush > host.realtime )
		finalpush = host.realtime;

	frame = NULL;

	for( frame2 = NULL, i = 0; i < SV_UPDATE_BACKUP; i++, frame2 = frame )
	{
		frame = &cl->frames[(cl->netchan.outgoing_sequence - 1) & SV_UPDATE_MASK];

		for( j = 0; j < frame->num_entities; j++ )
		{
			state = &svs.packet_entities[(frame->first_entity+j)%svs.num_client_entities];

			if( state->number <= 0 || state->number >= sv_maxclients->integer )
				continue;

			// you know, jge doesn't sound right given how the indexing normally works.

			lerp = &svgame.interp[state->number-1];

			if( lerp->nointerp )
				continue;

			if( state->health <= 0 || ( state->effects & EF_NOINTERP ))
				lerp->nointerp = true;

			if( !lerp->firstframe )
				lerp->firstframe = true;
			else if( SV_UnlagCheckTeleport( state->origin, lerp->finalpos ))
				lerp->nointerp = true;

			VectorCopy( state->origin, lerp->finalpos );
		}

		if( finalpush > frame->senttime )
			break;
	}


	if( i == SV_UPDATE_BACKUP || finalpush - frame->senttime > 1.0 )
	{
		Q_memset( svgame.interp, 0, sizeof( svgame.interp ));
		has_update = false;
		return;
	}

	if( !frame2 )
	{
		frame2 = frame;
		lerpFrac = 0;
	}
	else
	{
		if( frame2->senttime - frame->senttime == 0.0 )
		{
			lerpFrac = 0;
		}
		else
		{
			lerpFrac = (finalpush - frame->senttime) / (frame2->senttime - frame->senttime);
			lerpFrac = bound( 0.0f, lerpFrac, 1.0f );
		}
	}

	for( i = 0; i < frame->num_entities; i++ )
	{
		state = &svs.packet_entities[(frame->first_entity+i)%svs.num_client_entities];

		if( state->number <= 0 || state->number >= sv_maxclients->integer )
			continue;

		clientnum = state->number - 1;
		check = &svs.clients[clientnum];

		if( check->state != cs_spawned || check == cl )
			continue;

		lerp = &svgame.interp[clientnum];

		if( !lerp->active || lerp->nointerp )
			continue;

		lerpstate = SV_FindEntInPack( state->number, frame2 );

		if( !lerpstate )
		{
			tempvec[0] = state->origin[0];
			tempvec[1] = state->origin[1];
			tempvec[2] = state->origin[2];
		}
		else
		{
			tempvec2[0] = lerpstate->origin[0] - state->origin[0];
			tempvec2[1] = lerpstate->origin[1] - state->origin[1];
			tempvec2[2] = lerpstate->origin[2] - state->origin[2];

			VectorMA( state->origin, lerpFrac, tempvec2, tempvec );
		}

		VectorCopy( tempvec, lerp->curpos );
		VectorCopy( tempvec, lerp->newpos );

		if( !VectorCompare( tempvec, check->edict->v.origin ))
		{
			VectorCopy( tempvec, check->edict->v.origin );
			SV_LinkEdict( check->edict, false );
			lerp->moving = true;
		}
	}
}

void SV_PreRunCmd( sv_client_t *cl, usercmd_t *ucmd, int random_seed )
{
	svgame.pmove->runfuncs = true;
	svgame.dllFuncs.pfnCmdStart( cl->edict, ucmd, random_seed );
}

/*
===========
SV_RunCmd
===========
*/
void SV_RunCmd( sv_client_t *cl, usercmd_t *ucmd, int random_seed )
{
	edict_t	*clent;
	vec3_t	oldvel;
	usercmd_t cmd;
	int	oldmsec;
   
	clent = cl->edict;

	if( cl->next_movetime > host.realtime )
	{
		cl->last_movetime += ( ucmd->msec * 0.001f );
		return;
	}

	cl->next_movetime = 0.0;

	// chop up very long commands
	if( ucmd->msec > 50 )
	{
		cmd = *ucmd;
		oldmsec = ucmd->msec;
		cmd.msec = oldmsec / 2;

		SV_RunCmd( cl, &cmd, random_seed );

		cmd.msec = oldmsec / 2 + (oldmsec & 1);	// give them back thier msec.
		cmd.impulse = 0;
		SV_RunCmd( cl, &cmd, random_seed );
		return;
	}

	PM_CheckMovingGround( clent, ucmd->msec * 0.001f );

	VectorCopy( clent->v.v_angle, svgame.pmove->oldangles ); // save oldangles
	if( !clent->v.fixangle ) VectorCopy( ucmd->viewangles, clent->v.v_angle );

	// copy player buttons
	clent->v.button = ucmd->buttons;
	if( ucmd->impulse ) clent->v.impulse = ucmd->impulse;

	if( ucmd->impulse == 204 )
	{
		// force client.dll update
		SV_RefreshUserinfo();
	}

	if( clent->v.flags & FL_DUCKING ) 
		SV_SetMinMaxSize( clent, svgame.pmove->player_mins[1], svgame.pmove->player_maxs[1] );
	else SV_SetMinMaxSize( clent, svgame.pmove->player_mins[0], svgame.pmove->player_maxs[0] );

	if( !cl->spectator )
	{
		svgame.globals->time = sv.time + host.frametime;
		svgame.dllFuncs.pfnPlayerPreThink( clent );
		SV_RunThink( clent ); // clients cannot be deleted from map

		// If conveyor, or think, set basevelocity, then send to client asap too.
		if( !VectorIsNull( clent->v.basevelocity ))
			VectorCopy( clent->v.basevelocity, clent->v.clbasevelocity );
	}

	if(( sv_maxclients->integer <= 1 ) && !CL_IsInGame( ) || ( clent->v.flags & FL_FROZEN ))
		ucmd->msec = 0; // pause

	// setup playermove state
	SV_SetupPMove( svgame.pmove, clent, ucmd, cl->physinfo );

	// motor!
	svgame.dllFuncs.pfnPM_Move( svgame.pmove, true );

	// copy results back to client
	SV_FinishPMove( svgame.pmove, clent );

	if( !cl->spectator )
	{
		int	i;
		edict_t	*touch;
			
		// link into place and touch triggers
		SV_LinkEdict( clent, true );
		VectorCopy( clent->v.velocity, oldvel ); // save velocity

		svgame.globals->time = sv.time + host.frametime;

		// touch other objects
		for( i = 0; i < svgame.pmove->numtouch; i++ )
		{
			if( i == MAX_PHYSENTS ) break;
			touch = EDICT_NUM( svgame.pmove->physents[svgame.pmove->touchindex[i].ent].info );
			if( touch == clent ) continue;

			VectorCopy( svgame.pmove->touchindex[i].deltavelocity, clent->v.velocity );

			if( touch->v.groupinfo && clent->v.groupinfo )
			{
				if(( svs.groupop == 0 && ( touch->v.groupinfo & clent->v.groupinfo )) == 0 ||
				(svs.groupop == 1 && (touch->v.groupinfo & clent->v.groupinfo) != 0 ))
					continue;
			}

			if( touch->v.solid != SOLID_NOT )
			{
				SV_CopyPmtraceToGlobal( &svgame.pmove->touchindex[i] );
				svgame.dllFuncs.pfnTouch( touch, clent );
			}

			if( clent->v.solid != SOLID_NOT )
			{
				SV_CopyPmtraceToGlobal( &svgame.pmove->touchindex[i] );
				svgame.dllFuncs.pfnTouch( clent, touch );
			}
		}

		// restore velocity
		VectorCopy( oldvel, clent->v.velocity );
	}

	svgame.pmove->numtouch = 0;
}

/*
===========
SV_PostRunCmd

Done after running a player command.
===========
*/
void SV_PostRunCmd( sv_client_t *cl )
{
	edict_t	*clent;

	clent = cl->edict;
	if( !clent || clent->free ) return;

	svgame.pmove->runfuncs = false;	// all next calls ignore footstep sounds
		
	// run post-think
	if( cl->spectator )
		svgame.dllFuncs.pfnSpectatorThink( clent );
	else svgame.dllFuncs.pfnPlayerPostThink( clent );

	// restore frametime
	svgame.globals->time = sv.time + host.frametime;
	svgame.globals->frametime = host.frametime;
	svgame.dllFuncs.pfnCmdEnd( cl->edict );
}                                                                   