/*	AMX Mod X script.

	UTSounds aka Unreal Tournament Sounds plugin by Lev.

	URL: http://forums.alliedmods.net/showthread.php?t=103073

	Info:
		This is an event announcer plugin.
		Supported events:
			first spawn;
			first kill;
			headshot;
			teamkill;
			kill from specific weapon;
			multikill ladder (each next kill within defined time interval);
			killing spree ladder (each level at defined number of kills);
			selfkill when on killing spree;
			end of killing spree.

	Features:
		announcements can be made via:
			hud (for initiator);
			chat (for all players except initiator);
			sound (for killer, victim and other players separately);
		fully customizable announcements via ini file;
		per map ini files supported (place these ini files in config\utsounds folder);
		per map prefix ini files supported (place these ini files in config\utsounds folder);
		unlimited number of announces, so any number of levels for multikill and killing streak ladders;
		furthermore unlimited killing spree levels support by using last announce (it will be repeated each count of frags specified for it);
		any number of announces for an event, announce will be selected by random;
		each player can change these settings via configuration menu (they will be stored in user info field "_utsounds" in his config, so they will be taken in the account on each server with this plugin):
			switch all announcements off or on;
			switch particular announcement off or on;
			enable/disable announce destinations: hud, chat, sounds;
			hud announcement position: center, left, right, top, bottom.

	Chat command:
		/utsounds		- display info about plugin and opens configuration menu.
		/utsounds on	- enable announcements for this player (this setting is stored in user info field "_utsounds").
		/utsounds off	- disable announcements for this player (this setting is stored in user info field "_utsounds").

	Console command:
		ut_menu			- opens player's configuration menu.

	CVARs:
		ut_enable <0|1> (default=1)		// Enable/Disable whole plugin.
		ut_on <0|1> (default=1)			// On/Off for new players by default. They can use chat command /utsounds <on|off> to switch.
		ut_hudx <0|1|2> (default=1)		// Default X position of hud announce (0 - left, 1 - center, 2 - right)
		ut_hudy <0|1|2> (default=1)		// Default Y position of hud announce (0 - top, 1 - center, 2 - bottom)
		ut_test <0|1> (default=0)		// Enable output in chat name of weapon used to kill. Use this to get weapon name to set in "kill from specific weapon" event.

	INI file:
		There are 4 types of lines.
		Vertical line '|' is a splitter for different parts.

		Format for first spawn, first kill, headshot and killing spree over events:
			<identifier p|f|h|t|k|e>|<text for hud>|<message for chat>|<sound for initiator>[|<sound for victim>|<sound for all>]
		Format for multikill levels:
			<identifier m>|<number of kills>|<wait interval>|<text for hud>|<message for chat>|<sound for initiator>[|<sound for victim>|<sound for all>]
		Format for killing spree levels:
			<identifier s>|<number of kills>|<text for hud>|<message for chat>|<sound for initiator>[|<sound for victim>|<sound for all>]
		Format for kill from specific weapon:
			<identifier w>|<weapon name>|<text for hud>|<message for chat>|<sound for initiator>[|<sound for victim>|<sound for all>]

		Identifier:
			p - first spawn;
			f - first kill;
			h - headshot;
			t - teamkill;
			w - kill from specific weapon;
			m - level for multikill;
			s - level for killing spree;
			k - selfkill when on killing spree;
			e - for ending of killing spree.
		Number of kills: 
			number length in chars is limited to 4;
			this is number of kills for this level to be triggered for multikill and killing spree (integer, i.e. 10).
		Wait interval: 
			seconds that given to get each next kill from previous multikill level up to this level (float, i.e. 3.1).
		Weapon name:
			weapon name for example: crowbar, gauss;
			this name is game internal and can distinct from commonly known weapon name;
			to get weapon name use ut_test CVAR.
		Text for hud:
			it will be displayed for player who is activated an event (for killing spree over it will be displayed to victim).
		Message for chat:
			it will be displayed for all players except player to whom hud announce was shown;
		Tags:
			for messages for chat and texts for hud you can use tags to substitute players' names, frag count and weapon name;
			[] are imprescriptible part of a tag;
			only first occurence will be replaced (i.e. you can use each tag once);
			[UTS_K] will be replaced with killer name (for first spawn event this is used to substitute player name);
			[UTS_V] will be replaced with victim name;
			[UTS_W] will be replaced with weapon name;
			[UTS_C] will be replaced with number of frags (total for killing spree, and last contiguous for multikill);
			for killing spree end event when selfkill or teamkill [UTS_V] will be a name of player whose spree ended and [UTS_K] will be "himself".
		Sound file:
			<mod dir>\sound relative path (i.e. file <mod dir>\sound\custom\example.wav should be specified as custom/example.wav);
			you can specify three different files for each announcements one for initiator, one for victim and one for other players;
			if you wish use same files for them then use * to do that (i.e. custom/example.wav|*|*) or you can just do not put splitters and * (i.e. simply specify custom/example.wav);
			you can use * only for third file then it will be assumed the same as second sound;
			if you wish that no sound would be for a player(s) then leave field empty (example custom/example.wav||).

		Selfkill event triggered only if player is on killing spree.

		For multikill last level will be repeated each next frag made within interval specified for last level.

		For killing spree last level will be repeated each next <number> frags.
		For example you set 30 for last killing spree level. Then this announcement will be done at 30, 60, 90, etc frags.

		Don't want particular announcement? Comment it out.
		Don't want chat or hud messages? Leave them empty "||". Same for the sound.

		If you want to use plugin defaults for all announcements delete this ini file.

	Installation:
		put utsounds.amxx to plugins folder;
		add utsounds.amxx to config\plugins.ini file;
		put utsounds.ini in config folder (edit it if you want to customize announcements);
		put utsounds.txt in data\lang folder;
		extract sound.zip into your <mod dir>\sound folder.

	Notes:
		User info field "_utsounds" is used to store user's settings.

	Credits:
		Freejam for arousing my interest to write this plugin, beta testing and good suggestions.
		Preset KillStreak by Simon Logic for the settings system.
		Unreal Tournament Sounds by hleV for event handling.
		crazyeffect for help with MultiLanguage.

	Change Log:
		v1.0 [2009.09.09]
			Initial release.
		v1.1 [2009.09.09]
			! Fixed: ut_on CVAR behavior corrected;
			! Fixed: ut_enable CVAR behavior corrected;
			! Fixed: killing spree over event in case of selfkill: chat message corrected.
		v1.2 [2009.09.09]
			! Fixed: ut_on CVAR behavior corrected a little bit more. Finally :).
		v1.3 [2009.09.10]
			! Changed: small code optimizations.
		v1.4 [2009.09.10]
			! Fixed: now multikill intervals works as they should be (time is counted to reach this level, not next);
			! Changed: removed CVARs to disable announces (if you don't wish particular announcement then comment it out in the ini file);
			! Changed: now defaults will be loaded only if ini file is not found (before defaults were loaded per event).
			! Changed: chat announces will be no longer displayed to whom hud announce was shown.
		v1.5 [2009.09.10]
			! Changed: changed version cvar to "utsounds_version" (forget to change it before).
		v1.6 [2009.09.18]
			! Changed: added MultiLanguage to info and status messages. ("Thanks" to crazyeffect))
		v1.7 [2009.11.26]
			! Fixed: bug with multilanguage fixed (for example miscstats work now will not be interfered).
			! Fixed: player enumeration fixed.
			! Changed: small code optimizations.
			! Changed: dynamic arrays are used to store announcements.
			! Changed: multiple announces for an event selected by random are supported.
			! Changed: unlimited number of announces are supported.
			! Changed: ability to specify different sounds for killer, victim and others players.
			! Changed: fully custom messages for hud and chat (used tags instead of % arguments).
			! Changed: number of kill for each multikill level specified in config file.
			! Changed: ut_multikill CVAR deleted.
			! Added: new event type: kill with specific weapon.
			! Added: new settings are saved for user: hud position and enable/disable of specific events.
			! Changed: added few sounds.
			! Changed: ini file format.
			! Changed: multilanguage dictionary file.
		v1.8 [2009.11.26]
			! Added: configuration menu.
		v1.9 [2009.11.28]
			! Changed: texts corrections.
		v2.0 [2009.11.29]
			! Fixed: crash when announce count was zero.
			! Added: new settings are saved for user: enable/disable announce destinations: hud, chat and sounds.
			! Changed: multilanguage dictionary file.
		v2.1 [2009.12.04]
			! Fixed: weapon name detection in CS.
			! Added: new announce type: team kill.
			! Changed: multilanguage dictionary file.
			! Changed: how announce types are stored for user.
			! Fixed: bug with tags replacements when player name is too long.

	Todo:
		Multiple multikill ladders or bunch kill events.

*/

#pragma semicolon 1
#pragma ctrlchar '\'

#include <amxmodx>
#include <amxmisc>

#define AUTHOR "Lev"
#define PLUGIN "UTSounds"
#define VERSION "2.1"
#define VERSION_CVAR "utsounds_version"

// System constants
//#define _DEBUG		// Enable debug output at server console.
//#define _DEBUG_CONFIG	// Enable debug of config loading.

// Absolute base hud positions
#define HUD_X0		0.0
#define HUD_X1		-1.0
#define HUD_X2		1.0
#define HUD_Y0		0.1
#define HUD_Y1		0.5
#define HUD_Y2		0.90
// Relative hud positions
#define HUD_PLAY_Y	-0.08
#define HUD_FK_Y	-0.08
#define HUD_HS_Y	-0.05
#define HUD_SK_Y	-0.02
#define HUD_TK_Y	-0.02
#define HUD_MK_Y	-0.02
#define HUD_KS_Y	0.01
#define HUD_KSE_Y	0.01
#define HUD_WK_Y	0.04

// Constants
#define TASK_SHOWINFO_BASE			100
#define TASK_PLAYANNOUNCE_BASE		200
#define DELAY_BEFORE_INFO			5.0	// delay before informing player about UTSounds in seconds.
#define DELAY_BEFORE_PLAY_ANNOUNCE	0.5	// delay before announce first spawn event.

// MAX constants
#define MAX_PLAYERS				32
#define MAX_EVENT_TYPES			9
#define MAX_WEAPONNAME_LENGTH	32
#define MAX_PLAYERNAME_LENGTH	32
#define MAX_NUMBER_LENGTH		4
#define MAX_MSG_LENGTH			63
#define MAX_SND_LENGTH			63
#define MAX_MULTIKILL_LEVELS	10
#define MAX_KILLSPREE_LEVELS	10
#define MAX_FILE_LINE_LENGTH	2 + MAX_WEAPONNAME_LENGTH + MAX_MSG_LENGTH * 2 + MAX_SND_LENGTH * 3 + 6

// Temporary storage for different needs
new _temp[MAX_FILE_LINE_LENGTH];

// Constants
new const 
	_configFileName[] = "utsounds.ini",
	_pluginConfigDir[] = "utsounds",
	_soundDir[] = "sound",
	_clientInfoField[] = "_utsounds",
	_splitter = '|';

// CVARs
new pcvar_ut_enable;
new pcvar_ut_on;
new pcvar_ut_test;
new pcvar_ut_hudx;
new pcvar_ut_hudy;

// Plugin data
new _coloredMenus;

// Players' data
new bool:_firstKill = true;										// First kill on map
new bool:_enabledForPlayer[MAX_PLAYERS + 1] = { true, ... };	// Is UTSounds enabled for a player
new bool:_firstSpawn[MAX_PLAYERS + 1] = { true, ... };			// Is first spwan for a player
new bool:_onKillSpree[MAX_PLAYERS + 1] = { false, ... };		// Is player currently on killing spree
new bool:_enabledHud[MAX_PLAYERS + 1] = { true, ... };			// Is hud announce enabled
new bool:_enabledChat[MAX_PLAYERS + 1] = { true, ... };			// Is chat announce enabled
new bool:_enabledSound[MAX_PLAYERS + 1] = { true, ... };		// Is sound announce enabled
new _multiKillKills[MAX_PLAYERS + 1] = { 0, ... };				// Current number of contiguous kills for multikill for a player
new _killSpreeKills[MAX_PLAYERS + 1] = { 0, ... };				// Contigious number of kills for a player
new _hudX[MAX_PLAYERS + 1] = { 1, ... };						// X position of hud announce (0 - left, 1 - center, 2 - right)
new _hudY[MAX_PLAYERS + 1] = { 1, ... };						// Y position of hud announce (0 - top, 1 - center, 2 - bottom)
new _events[MAX_PLAYERS + 1][MAX_EVENT_TYPES + 1];				// Will store string for each player containing enabled events

// Events
new Array:_eventFirstSpawn;
new Array:_eventFirstKill;
new Array:_eventHeadshot;
new Array:_eventSelfKill;
new Array:_eventTeamKill;
new Array:_eventKillSpreeEnd;
new _eventWeaponsCount;
new Array:_eventWeaponsNames;		// This will contain names of weapons
new Array:_eventWeaponsAnnounces;	// This will contain pointer to dynamic array that will hold announce numbers
new _eventMultiKillLevelsCount;
new Array:_eventMultiKillLevels;	// This will contain numbers of kill for a level
new Array:_eventMultiKillIntervals;	// This will contain interval for a level
new Array:_eventMultiKillAnnounces;	// This will contain pointer to dynamic array that will hold announce numbers
new _eventKillSpreeLevelsCount;
new Array:_eventKillSpreeLevels;	// This will contain numbers of kill for a level
new Array:_eventKillSpreeAnnounces;	// This will contain pointer to dynamic array that will hold announce numbers
// Announces
new _announcesCount;
new Array:_announceHud;
new Array:_announceChat;
new Array:_announceSndInitiator;
new Array:_announceSndVictim;
new Array:_announceSndOthers;

// Default announcements
new const _defaultAnnouncements[][] = {
	"p||[UTS_K] has joined the game|misc/prepare3.wav",
	"f|First Blood!|[UTS_K] drew first blood|misc/firstblood.wav",
	"h|Headshot!|[UTS_K] headshot killed [UTS_V] with [UTS_W]|misc/headshot.wav",
	"k|Holyshit!|[UTS_K] killed himself|misc/holyshit.wav",
	"t|Holyshit!|[UTS_K] killed his teammate [UTS_V]|misc/holyshit.wav",
	"e|Killing spree ended!|[UTS_V]'s killing spree was ended by [UTS_K]|misc/strangecrap.wav",
	//";w|crowbar|Humiliation!|[UTS_K] killed [UTS_V] with [UTS_W]|misc/humiliation.wav",
	"w|crowbar|Humiliation!|[UTS_K] killed [UTS_V] with [UTS_W]|misc/ky1.wav",
	"w|crowbar|Humiliation!|[UTS_K] killed [UTS_V] with [UTS_W]|misc/ky2.wav",
	"w|crowbar|Humiliation!|[UTS_K] killed [UTS_V] with [UTS_W]|misc/ky3.wav",
	"m|2|4.5|Double Kill!|[UTS_K] double killing|misc/doublekill.wav",
	"m|3|4.0|Triple Kill!|[UTS_K] triple killing|misc/triplekill.wav",
	"m|4|3.5|Multi Kill!|[UTS_K] multi killing|misc/multikill.wav",
	"m|5|3.0|Mega Kill!!|[UTS_K] mega killing|misc/megakill.wav",
	"m|6|2.5|ULTRA KILL!!|[UTS_K] ultra killing|misc/ultrakill.wav",
	"m|7|2.0|M O N S T E R   K I L L !!!|[UTS_K] MONSTER killing|misc/monsterkill.wav",
	//";m|8|1.0|L U D I C R O U S  K I L L !!!|[UTS_K] LUDICROUS KILLING!|misc/ludicrouskill.wav",
	"s|5|Killing Spree!|[UTS_K] is on a killing spree!|misc/killingspree.wav",
	"s|10|Rampage!|[UTS_K] is on a rampage!|misc/rampage.wav",
	"s|20|Dominating!|[UTS_K] is dominating!|misc/dominating.wav",
	"s|30|Unstoppable!|[UTS_K] is unstoppable!|misc/unstoppable.wav",
	"s|50|GODLIKE!|[UTS_K] is Godlike!|misc/godlike.wav",
	"s|90|WICKED SICK!|WICKED SICK!!! [UTS_K] is on a killing spree with [UTS_C] frags!|misc/wickedsick.wav"
};

public plugin_init() 
{
	register_plugin(PLUGIN, VERSION, AUTHOR);
	register_cvar(VERSION_CVAR, VERSION, FCVAR_SPONLY | FCVAR_SERVER | FCVAR_UNLOGGED);
	register_dictionary("common.txt");
	register_dictionary("utsounds.txt");

	pcvar_ut_enable = register_cvar("ut_enable", "1");	// Enable/Disable whole plugin.
	pcvar_ut_on = register_cvar("ut_on", "1");			// On/Off for new players by default. They can use command /utsounds <on|off> to switch.
	pcvar_ut_test = register_cvar("ut_test", "0");		// Enable output in chat name of weapon used to kill.
	pcvar_ut_hudx = register_cvar("ut_hudx", "1");		// Default X position of hud announce (0 - left, 1 - center, 2 - right)
	pcvar_ut_hudy = register_cvar("ut_hudy", "1");		// Default Y position of hud announce (0 - top, 1 - center, 2 - bottom)

	register_event("ResetHUD", "OnPlayerSpawn", "be");
	register_event("DeathMsg", "OnDeath", "a");

	register_clcmd("say", "CmdSay");
	register_clcmd("say_team", "CmdSay");

	register_clcmd("ut_menu", "CmdMenu", ADMIN_ALL, "- displays UTSounds menu");
	register_menu("UTSounds Menu", 1023, "ActionMainMenu");
	register_menu("Announcements Menu", 1023, "ActionAnnouncementsMenu");
	_coloredMenus = colored_menus();
}



//****************************************
//*                                      *
//*  Menu                                *
//*                                      *
//****************************************

public CmdMenu(id, level, cid)
{
	ShowMainMenu(id);
	return PLUGIN_HANDLED;
}

ShowMainMenu(id)
{
	if (!get_pcvar_num(pcvar_ut_enable))
		return PLUGIN_HANDLED;

	// Build menu
	new menuBody[512];
	new len = format(menuBody, charsmax(menuBody), (_coloredMenus ? "\\y%L\\w\n\n" : "%L\n\n"), id, "UTS_MENU");
	len += format(menuBody[len], charsmax(menuBody) - len, (_coloredMenus ? "1. %L\\R\\r%L\\w\n" : "1. %L: %L\n"), id, "UTS_STATUS", id, _enabledForPlayer[id] ? "ON" : "OFF");
	len += format(menuBody[len], charsmax(menuBody) - len, (_coloredMenus ? "2. %L\\R\\r%L\\w\n" : "2. %L: %L\n"), id, "UTS_HUD", id, _enabledHud[id] ? "ON" : "OFF");
	len += format(menuBody[len], charsmax(menuBody) - len, (_coloredMenus ? "3. %L\\R\\r%L\\w\n" : "3. %L: %L\n"), id, "UTS_CHAT", id, _enabledChat[id] ? "ON" : "OFF");
	len += format(menuBody[len], charsmax(menuBody) - len, (_coloredMenus ? "4. %L\\R\\r%L\\w\n" : "4. %L: %L\n"), id, "UTS_SOUND", id, _enabledSound[id] ? "ON" : "OFF");
	len += format(menuBody[len], charsmax(menuBody) - len, (_coloredMenus ? "5. %L\n" : "5. %L\n"), id, "UTS_CONF_ANNOUNCES");
	len += format(menuBody[len], charsmax(menuBody) - len, (_coloredMenus ? "6. %L\\R\\r%L\\w\n" : "5. %L: %L\n"), id, "UTS_HUD_POS", "X", id, _hudX[id] == 1 ? "UTS_CENTER" : _hudX[id] < 1 ? "UTS_LEFT" : "UTS_RIGHT");
	len += format(menuBody[len], charsmax(menuBody) - len, (_coloredMenus ? "7. %L\\R\\r%L\\w\n" : "6. %L: %L\n"), id, "UTS_HUD_POS", "Y", id, _hudY[id] == 1 ? "UTS_CENTER" : _hudY[id] < 1 ? "UTS_TOP" : "UTS_BOTTOM");
	len += format(menuBody[len], charsmax(menuBody) - len, "\n8. %L\n", id, "UTS_SAVE");
	format(menuBody[len], charsmax(menuBody) - len, "0. %L", id, "EXIT");

	// Show menu
	new keys = MENU_KEY_1|MENU_KEY_2|MENU_KEY_3|MENU_KEY_4|MENU_KEY_5|MENU_KEY_6|MENU_KEY_7|MENU_KEY_8|MENU_KEY_0;
	show_menu(id, keys, menuBody, -1, "UTSounds Menu");

	return PLUGIN_HANDLED;
}

public ActionMainMenu(id, key)
{
	switch(key + 1)
	{
		case 1:
		{
			_enabledForPlayer[id] = !_enabledForPlayer[id];
			ShowMainMenu(id);
		}
		case 2:
		{
			_enabledHud[id] = !_enabledHud[id];
			ShowMainMenu(id);
		}
		case 3:
		{
			_enabledChat[id] = !_enabledChat[id];
			ShowMainMenu(id);
		}
		case 4:
		{
			_enabledSound[id] = !_enabledSound[id];
			ShowMainMenu(id);
		}
		case 5:
		{
			ShowAnnouncementsMenu(id);
		}
		case 6:
		{
			_hudX[id]++;
			if (_hudX[id] > 2) _hudX[id] = 0;
			ShowMainMenu(id);
		}
		case 7:
		{
			_hudY[id]++;
			if (_hudY[id] > 2) _hudY[id] = 0;
			ShowMainMenu(id);
		}
		case 8:
		{
			// Save settings and exit
			SaveSettings(id);
			client_print(id, print_chat, "[UTSounds] %L", id, "UTS_SAVED");
			ShowMainMenu(id);
		}
		case 10:
		{
			// Exit without saving
		}
	}

	return PLUGIN_HANDLED;
}

ShowAnnouncementsMenu(id)
{
	if (!get_pcvar_num(pcvar_ut_enable))
		return PLUGIN_HANDLED;

	// Build menu
	new menuBody[512];
	new len = format(menuBody, charsmax(menuBody), (_coloredMenus ? "\\y%L\\w\n\n" : "%L\n\n"), id, "UTS_ANNOUNCES_MENU");
	len += format(menuBody[len], charsmax(menuBody) - len, (_coloredMenus ? "1. %L\\R\\r%L\\w\n" : "1. %L: %L\n"), id, "UTS_FIRST_SPAWN", id, FindCharInString(_events[id], 'P') ? "OFF" : "ON");
	len += format(menuBody[len], charsmax(menuBody) - len, (_coloredMenus ? "2. %L\\R\\r%L\\w\n" : "2. %L: %L\n"), id, "UTS_FIRST_KILL", id, FindCharInString(_events[id], 'F') ? "OFF" : "ON");
	len += format(menuBody[len], charsmax(menuBody) - len, (_coloredMenus ? "3. %L\\R\\r%L\\w\n" : "3. %L: %L\n"), id, "UTS_HEADSHOT", id, FindCharInString(_events[id], 'H') ? "OFF" : "ON");
	len += format(menuBody[len], charsmax(menuBody) - len, (_coloredMenus ? "4. %L\\R\\r%L\\w\n" : "4. %L: %L\n"), id, "UTS_TEAMKILL", id, FindCharInString(_events[id], 'T') ? "OFF" : "ON");
	len += format(menuBody[len], charsmax(menuBody) - len, (_coloredMenus ? "5. %L\\R\\r%L\\w\n" : "5. %L: %L\n"), id, "UTS_WEAPON", id, FindCharInString(_events[id], 'W') ? "OFF" : "ON");
	len += format(menuBody[len], charsmax(menuBody) - len, (_coloredMenus ? "6. %L\\R\\r%L\\w\n" : "6. %L: %L\n"), id, "UTS_MULTIKILL", id, FindCharInString(_events[id], 'M') ? "OFF" : "ON");
	len += format(menuBody[len], charsmax(menuBody) - len, (_coloredMenus ? "7. %L\\R\\r%L\\w\n" : "7. %L: %L\n"), id, "UTS_KILL_STREAK", id, FindCharInString(_events[id], 'S') ? "OFF" : "ON");
	format(menuBody[len], charsmax(menuBody) - len, "\n0. %L", id, "BACK");

	// Show menu
	new keys = MENU_KEY_1|MENU_KEY_2|MENU_KEY_3|MENU_KEY_4|MENU_KEY_5|MENU_KEY_6|MENU_KEY_7|MENU_KEY_0;
	show_menu(id, keys, menuBody, -1, "Announcements Menu");

	return PLUGIN_HANDLED;
}

public ActionAnnouncementsMenu(id, key)
{
	switch(key + 1)
	{
		case 1:
		{
			RemoveCharFromString(_events[id], 'p');
			if (FindCharInString(_events[id], 'P'))
				RemoveCharFromString(_events[id], 'P');
			else
				AddCharToString(_events[id], MAX_EVENT_TYPES, 'P');
			ShowAnnouncementsMenu(id);
		}
		case 2:
		{
			RemoveCharFromString(_events[id], 'f');
			if (FindCharInString(_events[id], 'F'))
				RemoveCharFromString(_events[id], 'F');
			else
				AddCharToString(_events[id], MAX_EVENT_TYPES, 'F');
			ShowAnnouncementsMenu(id);
		}
		case 3:
		{
			RemoveCharFromString(_events[id], 'h');
			if (FindCharInString(_events[id], 'H'))
				RemoveCharFromString(_events[id], 'H');
			else
				AddCharToString(_events[id], MAX_EVENT_TYPES, 'H');
			ShowAnnouncementsMenu(id);
		}
		case 4:
		{
			RemoveCharFromString(_events[id], 't');
			if (FindCharInString(_events[id], 'T'))
				RemoveCharFromString(_events[id], 'T');
			else
				AddCharToString(_events[id], MAX_EVENT_TYPES, 'T');
			ShowAnnouncementsMenu(id);
		}
		case 5:
		{
			RemoveCharFromString(_events[id], 'w');
			if (FindCharInString(_events[id], 'W'))
				RemoveCharFromString(_events[id], 'W');
			else
				AddCharToString(_events[id], MAX_EVENT_TYPES, 'W');
			ShowAnnouncementsMenu(id);
		}
		case 6:
		{
			RemoveCharFromString(_events[id], 'm');
			if (FindCharInString(_events[id], 'M'))
				RemoveCharFromString(_events[id], 'M');
			else
				AddCharToString(_events[id], MAX_EVENT_TYPES, 'M');
			ShowAnnouncementsMenu(id);
		}
		case 7:
		{
			RemoveCharFromString(_events[id], 's');
			RemoveCharFromString(_events[id], 'e');
			RemoveCharFromString(_events[id], 'k');
			if (FindCharInString(_events[id], 'S'))
			{
				RemoveCharFromString(_events[id], 'S');
				RemoveCharFromString(_events[id], 'E');
				RemoveCharFromString(_events[id], 'K');
			}
			else
			{
				AddCharToString(_events[id], MAX_EVENT_TYPES, 'S');
				AddCharToString(_events[id], MAX_EVENT_TYPES, 'E');
				AddCharToString(_events[id], MAX_EVENT_TYPES, 'K');
			}
			ShowAnnouncementsMenu(id);
		}
		case 10:
		{
			// Return back to main menu
			ShowMainMenu(id);
		}
	}

	return PLUGIN_HANDLED;
}



//****************************************
//*                                      *
//*  Commands                            *
//*                                      *
//****************************************

public CmdSay(id)
{
	if (!get_pcvar_num(pcvar_ut_enable))
		return PLUGIN_CONTINUE;

	new argc = read_argc();
	if (argc < 2) return PLUGIN_CONTINUE;

	read_argv(1, _temp, charsmax(_temp));

	if (!equali(_temp, "/utsounds", 9)) return PLUGIN_CONTINUE;

	if (argc == 2)
	{
		new arg1[10], arg2[4];
		parse(_temp, arg1, charsmax(arg1), arg2, charsmax(arg2));
		if (equali(arg2, "on")) return CmdOn(id);
		if (equali(arg2, "off")) return CmdOff(id);
		client_print(id, print_chat, "[UTSounds] %L", id, "UTS_INFO", VERSION);
		ShowInfo(TASK_SHOWINFO_BASE + id);
		ShowMainMenu(id);
	}
	else
	{
		read_argv(2, _temp, charsmax(_temp));
		if (equali(_temp, "on")) return CmdOn(id);
		if (equali(_temp, "off")) return CmdOff(id);
	}

	return PLUGIN_CONTINUE;
}

public CmdOn(id)
{
	if (!get_pcvar_num(pcvar_ut_enable))
		return PLUGIN_CONTINUE;

	_enabledForPlayer[id] = true;
	SaveSettings(id);
	ShowInfo(id + TASK_SHOWINFO_BASE);
	return PLUGIN_CONTINUE;
}

public CmdOff(id)
{
	if (!get_pcvar_num(pcvar_ut_enable))
		return PLUGIN_CONTINUE;

	_enabledForPlayer[id] = false;
	SaveSettings(id);
	ShowInfo(TASK_SHOWINFO_BASE + id);
	return PLUGIN_CONTINUE;
}

public ShowInfo(id)
{
	id -= TASK_SHOWINFO_BASE;
	if (id < 1 || id > MAX_PLAYERS) return;

	if (_enabledForPlayer[id])
		client_print(id, print_chat, "[UTSounds] %L", id, "UTS_ON");
	else
		client_print(id, print_chat, "[UTSounds] %L", id, "UTS_OFF");
}



//****************************************
//*                                      *
//*  Connection                          *
//*                                      *
//****************************************

public client_connect(id) 
{
	_enabledForPlayer[id] = get_pcvar_num(pcvar_ut_on) > 0;
	_firstSpawn[id] = true;
	_onKillSpree[id] = false;
	_enabledHud[id] = true;
	_enabledChat[id] = true;
	_enabledSound[id] = true;
	_multiKillKills[id] = 0;
	_killSpreeKills[id] = 0;
	_hudX[id] = get_pcvar_num(pcvar_ut_hudx);
	_hudY[id] = get_pcvar_num(pcvar_ut_hudy);
	_events[id] = "";

	LoadSettings(id);

	if (get_pcvar_num(pcvar_ut_enable))
		set_task(DELAY_BEFORE_INFO, "ShowInfo", TASK_SHOWINFO_BASE + id);

	return PLUGIN_CONTINUE;
}

public client_disconnect(id) 
{
	_enabledForPlayer[id] = true;
	_firstSpawn[id] = true;
	_onKillSpree[id] = false;
	_enabledHud[id] = true;
	_enabledChat[id] = true;
	_enabledSound[id] = true;
	_multiKillKills[id] = 0;
	_killSpreeKills[id] = 0;
	_hudX[id] = get_pcvar_num(pcvar_ut_hudx);
	_hudY[id] = get_pcvar_num(pcvar_ut_hudy);
	_events[id] = "";
}



//****************************************
//*                                      *
//*  Settings                            *
//*                                      *
//****************************************

LoadSettings(id)
{
	get_user_info(id, _clientInfoField, _temp, charsmax(_temp));
#if defined _DEBUG
	server_print("id: %i, clientInfo: %s", id, _temp);
#endif
	// Read player settings
	if (_temp[0] != 0)
	{
		_enabledForPlayer[id] = _temp[0] != '0';
		// Read hud position settings for player
		if (_temp[1] >= '0' && _temp[1] <= '2')
		{
			_hudX[id] = _temp[1] - 0x30;
			if (_temp[2] >= '0' && _temp[2] <= '2')
			{
				_hudY[id] = _temp[2] - 0x30;
				// Read event settings for player
				if (_temp[3] != 0)
				{
					strtok(_temp[3], _events[id], MAX_EVENT_TYPES, _temp[0], charsmax(_temp), '|');
					// Read announce destinations.
					if (_temp[0] != 0)
					{
						new dest = _temp[0] - 0x30;
						_enabledHud[id] = dest & (1 << 0) > 0;
						_enabledChat[id] = dest & (1 << 1) > 0;
						_enabledSound[id] = dest & (1 << 2) > 0;
					}
				}
			}
		}
	}
#if defined _DEBUG
	server_print("_enabledForPlayer: %i, _hudX: %i, _hudY: %i, _events: %s", _enabledForPlayer[id], _hudX[id], _hudY[id], _events[id]);
	server_print("_enabledHud: %i, _enabledChat: %i, _enabledSound: %i", _enabledHud[id], _enabledChat[id], _enabledSound[id]);
#endif
}

SaveSettings(id)
{
	client_cmd(id, "setinfo \"%s\" \"%i%i%i%s|%i\"", _clientInfoField, _enabledForPlayer[id], _hudX[id], _hudY[id], _events[id], 
		(_enabledHud[id] ? (1 << 0) : 0) + (_enabledChat[id] ? (1 << 1) : 0) + (_enabledSound[id] ? (1 << 2) : 0));
}



//****************************************
//*                                      *
//*  First Spawn                         *
//*                                      *
//****************************************

public OnPlayerSpawn(id)
{
	if (!_firstSpawn[id] || !get_pcvar_num(pcvar_ut_enable) || !is_user_alive(id))
		return PLUGIN_CONTINUE;

	set_task(DELAY_BEFORE_PLAY_ANNOUNCE, "PlayerSpawnTask", TASK_PLAYANNOUNCE_BASE + id);
	_firstSpawn[id] = false;
	return PLUGIN_CONTINUE;
}

public PlayerSpawnTask(id)
{
	id -= TASK_PLAYANNOUNCE_BASE;
	if (id < 1 || id > MAX_PLAYERS) return;

	MakeAnnounce('p', id, 0);
}



//****************************************
//*                                      *
//*  Death Event                         *
//*                                      *
//****************************************

public OnDeath() 
{
	if (!get_pcvar_num(pcvar_ut_enable))
	{
		_firstKill = false;
		return PLUGIN_CONTINUE;
	}

	new killer = read_data(1);
	new victim = read_data(2);
#if defined _DEBUG
	server_print("killer: %i, victim: %i", killer, victim);
#endif

	// Return if there is no victim
	if (!victim) return PLUGIN_CONTINUE;
	// If no killer then this is selfkill
	if (!killer) killer = victim;


	// Get weapon name
	new weaponName[32];
	new num = read_datanum();
#if defined _DEBUG
	server_print("Event data, count: %i", num);
	for (new i = 0; i < num; i++)
	{
		read_data(i, weaponName, charsmax(weaponName));
		server_print("%i: \"%s\"", i, weaponName);
	}
#endif
	// In HL weapon name is in 4 parameter, in CS it is 5
	read_data(num - 1, weaponName, charsmax(weaponName));
#if defined _DEBUG
	server_print("weaponName: %s", weaponName);
#endif
	// Output weapon name in killer's chat for testing purposes
	if (get_pcvar_num(pcvar_ut_test))
		client_print(killer, print_chat, "Weapon name: %s", weaponName);


	// First kill
	if (_firstKill)
	{
		_firstKill = false;
		MakeAnnounce('f', killer, victim, weaponName);
	}


	// Headshot
	new weapon, hitBox;
	new agressor = get_user_attacker(victim, weapon, hitBox);
#if defined _DEBUG
	server_print("agressor: %i, weapon: %i, hitBox: %i", agressor, weapon, hitBox);
#endif
	if (hitBox == 1 && agressor == killer)
	{
		MakeAnnounce('h', killer, victim, weaponName);
	}


	// Weapon event
	if (_eventWeaponsCount > 0)
	{
		new pos = ArrayFindString(_eventWeaponsNames, weaponName);
		// If event for weapon is found
		if (pos >= 0)
		{
			MakeAnnounce('w', killer, victim, weaponName, 0, pos);
		}
	}


	// Selfkill
	if (killer == victim)
	{
		// End multikill for victim
		_multiKillKills[victim] = 0;
		remove_task(victim);

		// End killing spree for victim
		if (_onKillSpree[victim])
		{
			_onKillSpree[victim] = false;
			MakeAnnounce('k', killer, victim, weaponName);
			MakeAnnounce('e', killer, victim, weaponName, 0, 0, false);
		}
		_killSpreeKills[victim] = 0;

		return PLUGIN_CONTINUE;
	}


	// Teamkill
	new killerTeamStr[32], victimTeamStr[32];
#if defined _DEBUG
	new killerTeam = get_user_team(killer, killerTeamStr, charsmax(killerTeamStr));
	new victimTeam = get_user_team(victim, victimTeamStr, charsmax(victimTeamStr));
	server_print("killer: %i, team id: %i, team name: \"%s\"", killer, killerTeam, killerTeamStr);
	server_print("victim: %i, team id: %i, team name: \"%s\"", victim, victimTeam, victimTeamStr);
#else
	get_user_team(killer, killerTeamStr, charsmax(killerTeamStr));
	get_user_team(victim, victimTeamStr, charsmax(victimTeamStr));
#endif
	if (killerTeamStr[0] != 0 && equali(killerTeamStr, victimTeamStr))
	{
		MakeAnnounce('t', killer, victim, weaponName);

		// End multikill for killer
		_multiKillKills[killer] = 0;
		remove_task(killer);

		// End killing spree for killer
		if (_onKillSpree[killer])
		{
			_onKillSpree[killer] = false;
			MakeAnnounce('e', -1, killer, weaponName, 0, 0, false);
		}
		_killSpreeKills[killer] = 0;

		return PLUGIN_CONTINUE;
	}


	// Multikill
	if (_eventMultiKillLevelsCount > 0)
	{
		_multiKillKills[killer]++;
		// Seek for a triggered level
		new level = FindLevel(_eventMultiKillLevels, _multiKillKills[killer]);
#if defined _DEBUG
	server_print("killer: %i, multikills: %i, level: %i", killer, _multiKillKills[killer], level);
#endif
		// Get default interval
		new Float:interval = ArrayGetCell(_eventMultiKillIntervals, 0);
		// If level not found we will repeat last level
		if (level == -1)
			level = _eventMultiKillLevelsCount - 1;
#if defined _DEBUG
	server_print("level: %i", level);
#endif
		// If level is found
		if (level >= 0)
		{
			MakeAnnounce('m', killer, victim, weaponName, _multiKillKills[killer], level);

			if (level == _eventMultiKillLevelsCount - 1)
				interval = ArrayGetCell(_eventMultiKillIntervals, level);
			else
				interval = ArrayGetCell(_eventMultiKillIntervals, level + 1);
		}
		// If we are in the middle between levels
		if (level < -1)
		{
			level = -2 - level;
			interval = ArrayGetCell(_eventMultiKillIntervals, level);
		}
#if defined _DEBUG
	server_print("interval: %f", interval);
#endif
		// Reset next kill countdown timer
		remove_task(killer);
		set_task(interval, "StopMultiKills", killer);

		// End multikill for victim
		_multiKillKills[victim] = 0;
		remove_task(victim);
	}


	// Kill spree
	if (_eventKillSpreeLevelsCount > 0)
	{
		_killSpreeKills[killer]++;
		// Seek for a triggered level
		new level = FindLevel(_eventKillSpreeLevels, _killSpreeKills[killer]);
		// If level not found check if we can repeat last level
		if (level < 0)
		{
			new lastLevelKills = ArrayGetCell(_eventKillSpreeLevels, _eventKillSpreeLevelsCount - 1);
			if (_killSpreeKills[killer] - _killSpreeKills[killer] / lastLevelKills * lastLevelKills == 0)
				level = _eventKillSpreeLevelsCount - 1;
		}
		// If level is found
		if (level >= 0)
		{
			_onKillSpree[killer] = true;
			MakeAnnounce('s', killer, victim, weaponName, _killSpreeKills[killer], level);
		}

		// End killing spree for victim
		if (_onKillSpree[victim])
		{
			_onKillSpree[victim] = false;
			MakeAnnounce('e', killer, victim, weaponName, _killSpreeKills[victim], 0, level < 0);	// Sound only if no event for a spree for killer
		}
		_killSpreeKills[victim] = 0;
	}

	return PLUGIN_CONTINUE;
}

public StopMultiKills(id)
{
	_multiKillKills[id] = 0;
}



//****************************************
//*                                      *
//*  Announces                           *
//*                                      *
//****************************************

MakeAnnounce(announceType, killer, victim, const weaponName[] = "", count = 0, level = 0, bool:playSound = true)
{
#if defined _DEBUG
	server_print("killer: %i, victim: %i", killer, victim);
#endif
	new Array:announces;
	new announceNumber, r, g, b, Float:x, Float:y, initiator = killer;
	switch(announceType)
	{
		case 'p':
		{
			announces = _eventFirstSpawn;
			r = 255;
			y = HUD_PLAY_Y;
		}
		case 'f':
		{
			announces = _eventFirstKill;
			if (killer == victim) victim = -1;
			r = 255;
			y = HUD_FK_Y;
		}
		case 'h':
		{
			announces = _eventHeadshot;
			if (killer == victim) victim = -1;
			r = 255;
			y = HUD_HS_Y;
		}
		case 'k':
		{
			announces = _eventSelfKill;
			victim = -1;
			r = 255;
			y = HUD_SK_Y;
		}
		case 't':
		{
			announces = _eventTeamKill;
			r = 255;
			y = HUD_TK_Y;
		}
		case 'e':
		{
			announces = _eventKillSpreeEnd;
			initiator = victim;
			if (killer == victim) killer = -1;
			b = 255;
			y = HUD_KSE_Y;
		}
		case 'w':
		{
			if (level > _eventWeaponsCount) return;
			announces = ArrayGetCell(_eventWeaponsAnnounces, level);
			if (killer == victim) victim = -1;
			r = 255;
			g = 100;
			b = 55;
			y = HUD_WK_Y;
		}
		case 'm':
		{
			if (level > _eventMultiKillLevelsCount) return;
			announces = ArrayGetCell(_eventMultiKillAnnounces, level);
			r = 255;
			y = HUD_MK_Y;
		}
		case 's':
		{
			if (level > _eventKillSpreeLevelsCount) return;
			announces = ArrayGetCell(_eventKillSpreeAnnounces, level);
			b = 255;
			y = HUD_KS_Y;
		}
		default:
		{
			return;
		}
	}
	// Calculate hud positions
	switch(_hudX[initiator])
	{
		case 0: x = HUD_X0;
		case 1: x = HUD_X1;
		case 2: x = HUD_X2;
		default: x = HUD_X1;
	}
	switch(_hudY[initiator])
	{
		case 0: y += HUD_Y0;
		case 1: y += HUD_Y1;
		case 2: y += HUD_Y2;
		default: y += HUD_Y1;
	}

	// Make aim clear of hud texts
	if (_hudX[initiator] == 1 && _hudY[initiator] == 1 && y >= 0.48)
		y += 0.03;

	// Randomize announce
	new size = ArraySize(announces);
#if defined _DEBUG
	server_print("size: %i", size);
#endif
	if (size == 0)
		return;
	new rand = random(size);
#if defined _DEBUG
	server_print("rand: %i", rand);
#endif
	announceNumber = ArrayGetCell(announces, rand);
	DoAnnounce(announceType, announceNumber, initiator, killer, victim, weaponName, count, r, g, b, x, y, playSound);
}

DoAnnounce(announceType, announceNumber, initiator, killer, victim, const weaponName[], count, r, g, b, Float:x, Float:y, bool:playSound)
{
#if defined _DEBUG
	server_print("------------------------------------");
#endif
	new text[MAX_MSG_LENGTH + MAX_PLAYERNAME_LENGTH * 2 + MAX_WEAPONNAME_LENGTH + 1];
	new msg[MAX_MSG_LENGTH + MAX_PLAYERNAME_LENGTH * 2 + MAX_WEAPONNAME_LENGTH + 1];
	new snd1[MAX_SND_LENGTH + 1], snd2[MAX_SND_LENGTH + 1], snd3[MAX_SND_LENGTH + 1];

	// Get players' names
	new killerName[32], victimName[32];
	if (killer > 0)
		get_user_name(killer, killerName, charsmax(killerName));
	else if (killer == -1)
		killerName = "himself";
	if (victim > 0)
		get_user_name(victim, victimName, charsmax(victimName));
	else if (victim == -1)
		victimName = "himself";
#if defined _DEBUG
	server_print("ids: %i, %i, %i", initiator, killer, victim);
	server_print("names: %s, %s", killerName, victimName);
#endif

	// Prepare hud announce
	ArrayGetString(_announceHud, announceNumber, text, charsmax(text));
	FormatText(text, charsmax(text), killerName, victimName, weaponName, count);
#if defined _DEBUG
	server_print("hud: %s", text);
#endif
	// Prepare chat announce
	ArrayGetString(_announceChat, announceNumber, msg, charsmax(msg));
	FormatText(msg, charsmax(msg), killerName, victimName, weaponName, count);
#if defined _DEBUG
	server_print("chat: %s", msg);
#endif
	// Prepare sound announce
	ArrayGetString(_announceSndInitiator, announceNumber, snd1, charsmax(snd1));
	ArrayGetString(_announceSndVictim, announceNumber, snd2, charsmax(snd2));
	ArrayGetString(_announceSndOthers, announceNumber, snd3, charsmax(snd3));
	if (snd2[0] == '*') snd2 = snd1;
	if (snd3[0] == '*') snd3 = snd2;
#if defined _DEBUG
	server_print("%s", snd1);
	server_print("%s", snd2);
	server_print("%s", snd3);
#endif

	new players[MAX_PLAYERS], num;
	get_players(players, num);
	for (new i = 0; i < num; i++)
	{
		new id = players[i];
		if (_enabledForPlayer[id] && !is_user_bot(id) && !FindCharInString(_events[id], announceType - 32))
		{
			if (id == initiator)
			{
				// Hud announce
				if (text[0] != 0 && _enabledHud[id])
				{
					// If we are in hud resetting event, then nothing will be shown.
					set_hudmessage(r, g, b, x, y, 0, 6.0, 2.0, 0.01, 0.5, -1);
					show_hudmessage(id, text);
				}
			}
			else
			{
				// Chat announce
				if (msg[0] != 0 && _enabledChat[id])
					client_print(id, print_chat, msg);
			}
			// Sound announce
			if (playSound)
			{
				if (id == initiator)
				{
					if (snd1[0] != 0 && _enabledSound[id])
						client_cmd(id, "spk %s", snd1);
				}
				else if (id == victim)
				{
					if (snd2[0] != 0 && _enabledSound[id])
						client_cmd(id, "spk %s", snd2);
				}
				else 
				{
					if (snd3[0] != 0 && _enabledSound[id])
						client_cmd(id, "spk %s", snd3);
				}
			}
		}
	}
#if defined _DEBUG
	server_print("------------------------------------");
#endif
}



//****************************************
//*                                      *
//*  Format announce text                *
//*                                      *
//****************************************

FormatText(text[], maxlen, const killerName[], const victimName[], const weaponName[], count)
{
	new temp[MAX_MSG_LENGTH + 1];

	// For compatibility with old ini style
	if (containi(text, "%d") >= 0)
	{
		format(temp, charsmax(temp), text, killerName, count);
		copy(text, maxlen, temp);
	}
	else if (containi(text, "%s") >= 0)
	{
		format(temp, charsmax(temp), text, killerName, victimName);
		copy(text, maxlen, temp);
	}

	ReplaceWithString(text, maxlen, "[UTS_K]", killerName);
	ReplaceWithString(text, maxlen, "[UTS_V]", victimName);
	ReplaceWithString(text, maxlen, "[UTS_W]", weaponName);
	ReplaceWithNumber(text, maxlen, "[UTS_C]", count);
	return;
}



//****************************************
//*                                      *
//*  String functions                    *
//*                                      *
//****************************************

ReplaceWithString(text[], maxlen, const what[], const with[])
{
	replace(text, maxlen, what, "%s");
	copy(_temp, charsmax(_temp), text);
	format(text, maxlen, _temp, with);
}

ReplaceWithNumber(text[], maxlen, const what[], number)
{
	replace(text, maxlen, what, "%i");
	copy(_temp, charsmax(_temp), text);
	format(text, maxlen, _temp, number);
}

bool:FindCharInString(const string[], char1)
{
	new i = 0;
	while(string[i] != 0)
	{
		if (string[i] == char1)
			return true;
		i++;
	}
	return false;
}

RemoveCharFromString(string[], char1)
{
	new i = 0, pos = 0;
	while(string[i] != 0)
	{
		string[pos] = string[i];
		if (string[i] != char1)
			pos++;
		i++;
	}
	string[pos] = 0;
}

AddCharToString(string[], maxlen, char1)
{
	new len = strlen(string);
	if (len >= maxlen)
		return;
	string[len] = char1;
	string[len + 1] = 0;
}



//****************************************
//*                                      *
//*  Load ini file and precache          *
//*                                      *
//****************************************

public plugin_precache()
{
	// Create arrays for Events
	_eventFirstSpawn = ArrayCreate(1);
	_eventFirstKill = ArrayCreate(1);
	_eventHeadshot = ArrayCreate(1);
	_eventTeamKill = ArrayCreate(1);
	_eventSelfKill = ArrayCreate(1);
	_eventKillSpreeEnd = ArrayCreate(1);
	_eventWeaponsNames = ArrayCreate(MAX_WEAPONNAME_LENGTH + 1);
	_eventWeaponsAnnounces = ArrayCreate(1);
	_eventKillSpreeLevels = ArrayCreate(1);
	_eventKillSpreeAnnounces = ArrayCreate(1);
	_eventMultiKillLevels = ArrayCreate(1);
	_eventMultiKillIntervals = ArrayCreate(1);
	_eventMultiKillAnnounces = ArrayCreate(1);
	// Create arrays for Announces
	_announceHud = ArrayCreate(MAX_MSG_LENGTH + 1);
	_announceChat = ArrayCreate(MAX_MSG_LENGTH + 1);
	_announceSndInitiator = ArrayCreate(MAX_SND_LENGTH + 1);
	_announceSndVictim = ArrayCreate(MAX_SND_LENGTH + 1);
	_announceSndOthers = ArrayCreate(MAX_SND_LENGTH + 1);

	// Load configuration
	new bool:config = readConfigFile();
	// Load defaults if config not found.
	if (!config)
		for (new i = 0; i < sizeof(_defaultAnnouncements); i++)
			ParseLine(_defaultAnnouncements[i]);

	// Preacache sounds
	new snd[MAX_SND_LENGTH + 1], snd2[MAX_SND_LENGTH + 1];
	for (new i = 0; i < _announcesCount; i++)
	{
		ArrayGetString(_announceSndInitiator, i, snd, charsmax(snd));
		if (snd[0] != 0) precache_sound(snd);
		ArrayGetString(_announceSndVictim, i, snd2, charsmax(snd2));
		if (snd2[0] != '*' && snd2[0] != 0 && !equal(snd, snd2)) precache_sound(snd2);
		ArrayGetString(_announceSndOthers, i, snd, charsmax(snd));
		if (snd[0] != '*' && snd[0] != 0 && !equal(snd, snd2)) precache_sound(snd);
	}
}

bool:readConfigFile()
{
	new bool:isCustom;
	new i, txtLen;
	new configsDir[64], path[255];
	new buffer[MAX_FILE_LINE_LENGTH + 1];

	// Seek for per prefix and per map configs
	get_configsdir(configsDir, charsmax(configsDir));
	get_mapname(buffer, charsmax(buffer));
	formatex(path, charsmax(path), "%s/%s/%s.ini", configsDir, _pluginConfigDir, buffer);
	isCustom = bool:file_exists(path);
	if (!file_exists(path))
	{
		// get map prefix...
		i = contain(buffer, "_");
		if (i >= 0)
		{
			buffer[i + 1] = 0;
			formatex(path, charsmax(path), "%s/%s/%s.ini", configsDir, _pluginConfigDir, buffer);
			isCustom = bool:file_exists(path);
		}
		// Fall back to default config file
		if (!isCustom)
		{
			formatex(path, charsmax(path), "%s/%s", configsDir, _configFileName);
			// Return if no config files found
			if (!fileExists(path))
				return false;
		}
	}
	
	if (isCustom) log_amx("Using custom announcements from file: \"%s\"", path);

	// Parse config file
	i = 0;
	while (read_file(path, i, buffer, charsmax(buffer), txtLen))
	{
		i++;
		if (txtLen < 3) continue;
		ParseLine(buffer);
	}
	return true;
}

ParseLine(const data[])
{
	if (data[0] == 'p' && data[1] == _splitter) ArrayPushCell(_eventFirstSpawn, ParseAnnounce(data[2], true));
	if (data[0] == 'f' && data[1] == _splitter) ArrayPushCell(_eventFirstKill, ParseAnnounce(data[2], false));
	if (data[0] == 'h' && data[1] == _splitter) ArrayPushCell(_eventHeadshot, ParseAnnounce(data[2], false));
	if (data[0] == 'k' && data[1] == _splitter) ArrayPushCell(_eventSelfKill, ParseAnnounce(data[2], false));
	if (data[0] == 't' && data[1] == _splitter) ArrayPushCell(_eventTeamKill, ParseAnnounce(data[2], false));
	if (data[0] == 'e' && data[1] == _splitter) ArrayPushCell(_eventKillSpreeEnd, ParseAnnounce(data[2], false));
	if (data[0] == 'w' && data[1] == _splitter) AddWeapon(data[2]);
	if (data[0] == 'm' && data[1] == _splitter) AddMultiKill(data[2]);
	if (data[0] == 's' && data[1] == _splitter) AddKillSpree(data[2]);
}

AddWeapon(const data[])
{
	// Weapon
	new weaponName[MAX_WEAPONNAME_LENGTH + 1];
	strtok(data, weaponName, charsmax(weaponName), _temp, charsmax(_temp), _splitter);
	if (weaponName[0] == 0) return;
#if defined _DEBUG_CONFIG
	server_print("weaponName: %s", weaponName);
#endif

	// Find level or add new if not found
	new Array:announces;
	new pos = ArrayFindString(_eventWeaponsNames, weaponName);
#if defined _DEBUG_CONFIG
	server_print("pos %i", pos);
#endif
	if (pos == -1)
	{
		ArrayPushString(_eventWeaponsNames, weaponName);
		announces = ArrayCreate(1);
		ArrayPushCell(_eventWeaponsAnnounces, announces);
		_eventWeaponsCount++;
	}
	else if (pos < -1)
	{
		pos = -2 - pos;
		ArrayInsertStringBefore(_eventWeaponsNames, pos, weaponName);
		announces = ArrayCreate(1);
		ArrayInsertCellBefore(_eventWeaponsAnnounces, pos, announces);
		_eventWeaponsCount++;
	}
	else
	{
		announces = ArrayGetCell(_eventWeaponsAnnounces, pos);
	}

	// Parse announce
	ArrayPushCell(announces, ParseAnnounce(_temp, false));
}

AddMultiKill(const data[])
{
	// Kills count
	new number[MAX_NUMBER_LENGTH + 1];
	strtok(data, number, charsmax(number), _temp, charsmax(_temp), _splitter);
	new level = str_to_num(number);
	if (level < 1) return;
#if defined _DEBUG_CONFIG
	server_print("level %i", level);
#endif

	// Interval
	strtok(_temp, number, charsmax(number), _temp, charsmax(_temp), _splitter);
	new Float:interval = str_to_float(number);
	if (interval <= 0.0) return;
#if defined _DEBUG_CONFIG
	server_print("interval %f", interval);
#endif

	// Find level or add new if not found
	new Array:announces;
	new pos = FindLevel(_eventMultiKillLevels, level);
#if defined _DEBUG_CONFIG
	server_print("pos %i", pos);
#endif
	if (pos == -1)
	{
		ArrayPushCell(_eventMultiKillLevels, level);
		ArrayPushCell(_eventMultiKillIntervals, interval);
		announces = ArrayCreate(1);
		ArrayPushCell(_eventMultiKillAnnounces, announces);
		_eventMultiKillLevelsCount++;
	}
	else if (pos < -1)
	{
		pos = -2 - pos;
		ArrayInsertCellBefore(_eventMultiKillLevels, pos, level);
		ArrayInsertCellBefore(_eventMultiKillIntervals, pos, interval);
		announces = ArrayCreate(1);
		ArrayInsertCellBefore(_eventMultiKillAnnounces, pos, announces);
		_eventMultiKillLevelsCount++;
	}
	else
	{
		announces = ArrayGetCell(_eventMultiKillAnnounces, pos);
	}

	// Parse announce
	ArrayPushCell(announces, ParseAnnounce(_temp, false));
}

AddKillSpree(const data[])
{
	// Kills count
	new number[MAX_NUMBER_LENGTH + 1], level;
	strtok(data, number, charsmax(number), _temp, charsmax(_temp), _splitter);
	level = str_to_num(number);
	if (level < 1) return;
#if defined _DEBUG_CONFIG
	server_print("level %i", level);
#endif

	// Find level or add new if not found
	new Array:announces;
	new pos = FindLevel(_eventKillSpreeLevels, level);
#if defined _DEBUG_CONFIG
	server_print("pos %i", pos);
#endif
	if (pos == -1)
	{
		ArrayPushCell(_eventKillSpreeLevels, level);
		announces = ArrayCreate(1);
		ArrayPushCell(_eventKillSpreeAnnounces, announces);
		_eventKillSpreeLevelsCount++;
	}
	else if (pos < -1)
	{
		pos = -2 - pos;
		ArrayInsertCellBefore(_eventKillSpreeLevels, pos, level);
		announces = ArrayCreate(1);
		ArrayInsertCellBefore(_eventKillSpreeAnnounces, pos, announces);
		_eventKillSpreeLevelsCount++;
	}
	else
	{
		announces = ArrayGetCell(_eventKillSpreeAnnounces, pos);
	}

	// Parse announce
	ArrayPushCell(announces, ParseAnnounce(_temp, false));
}

/// Search for a string in array.
/// Returns position if exact match found (0 based).
/// Returns -1 if no exact match found.
ArrayFindString(Array:array, const string[])
{
#if defined _DEBUG_CONFIG
	server_print("string: \"%s\"", string);
#endif
	new stringInArray[MAX_WEAPONNAME_LENGTH + 1];
	new size = ArraySize(array);
	for (new i = 0; i < size; i++)
	{
		ArrayGetString(array, i, stringInArray, charsmax(stringInArray));
#if defined _DEBUG_CONFIG
	server_print("stringInArray: \"%s\"", stringInArray);
#endif
		if (equal(stringInArray, string)) return i;
	}
	return -1;
}

/// Search for a value in array.
/// Returns position if exact match found (0 based).
/// Returns -1 if no exact match found and all values are less than findLevel.
/// Returns position of value that is greater than findLevel in negative form (-2 based).
FindLevel(Array:array, findLevel)
{
#if defined _DEBUG_CONFIG
	server_print("findLevel: %i", findLevel);
#endif
	new level;
	new size = ArraySize(array);
	for (new i = 0; i < size; i++)
	{
		level = ArrayGetCell(array, i);
#if defined _DEBUG_CONFIG
	server_print("level: %i", level);
#endif
		if (level == findLevel) return i;
		else if (level > findLevel) return -2 - i;
	}
	return -1;
}



//****************************************
//*                                      *
//*  Parse announce data                 *
//*                                      *
//****************************************

/// Parses announce into dynamic arrays.
/// If isPersonal is true than no sound is parsed for victim and others.
/// Returns announce number.
ParseAnnounce(const data[], bool:isPersonal)
{
#if defined _DEBUG_CONFIG
	server_print("------------------------------------");
#endif
	new text[MAX_MSG_LENGTH + 1], snd[MAX_SND_LENGTH + 1];
	// Hud
	strtok(data, text, charsmax(text), _temp, charsmax(_temp), _splitter);
	trim(text);
	ArrayPushString(_announceHud, text);
#if defined _DEBUG_CONFIG
	server_print("%s", text);
#endif
	// Chat
	strtok(_temp, text, charsmax(text), _temp, charsmax(_temp), _splitter);
	trim(text);
	ArrayPushString(_announceChat, text);
#if defined _DEBUG_CONFIG
	server_print("%s", text);
#endif

	// Sound for initiator
	strtok(_temp, snd, charsmax(snd), _temp, charsmax(_temp), _splitter);
	trim(snd);
	if (snd[0] != 0 && !soundExists(snd))
		snd[0] = 0;
	ArrayPushString(_announceSndInitiator, snd);
#if defined _DEBUG_CONFIG
	server_print("%s, %s", snd, _temp);
#endif
	// If event is personal then no sounds for others
	if (isPersonal)
	{
		ArrayPushString(_announceSndVictim, "");
		ArrayPushString(_announceSndOthers, "");
		_announcesCount++;
		return _announcesCount - 1;
	}
	// Old ini style: Same sounds for victim and others
	trim(_temp);
	if (_temp[0] == 0)
	{
		ArrayPushString(_announceSndVictim, "*");
		ArrayPushString(_announceSndOthers, "*");
		_announcesCount++;
		return _announcesCount - 1;
	}

	// Sound for victim
	strtok(_temp, snd, charsmax(snd), _temp, charsmax(_temp), _splitter);
	trim(snd);
	if (snd[0] != '*' && snd[0] != 0 && !soundExists(snd))
		snd[0] = 0;
	ArrayPushString(_announceSndVictim, snd);
#if defined _DEBUG_CONFIG
	server_print("%s, %s", snd, _temp);
#endif
	// Sound for others
	strtok(_temp, snd, charsmax(snd), _temp, charsmax(_temp), _splitter);
	trim(snd);
	if (snd[0] != '*' && snd[0] != 0 && !soundExists(snd))
		snd[0] = 0;
	ArrayPushString(_announceSndOthers, snd);
#if defined _DEBUG_CONFIG
	server_print("%s", snd);
#endif

	_announcesCount++;
	return _announcesCount - 1;
}

stock bool:soundExists(const filename[])
{
	new file[MAX_SND_LENGTH + 1];
	formatex(file, charsmax(file), "%s/%s", _soundDir, filename);
	return fileExists(file);
}

stock bool:fileExists(const filename[])
{
	if (!file_exists(filename))
	{
		log_amx("File not found: \"%s\"", filename);
		return false;
	}
	return true;
}
