/*================================================================================
	
	------------------
	-*- Ping Faker -*-
	------------------
	
	---------------
	- Description -
	---------------
	This plugin can fake the display of a player's latency (ping) shown on
	the scoreboard. Unlike the "fakelag" command, it does not affect the
	player's real latency in any way.
	
	You can have all players report the same ping, or only fake it for those
	who have a specific admin flag. This last feature is especially useful
	when running a dedicated server from your own computer, when you don't
	want people to guess you're an admin/owner by looking at your low ping.

	
	---------
	- CVARS -
	---------
	pingfaker_mode 1	// 0 - disable faking, 1 - fake no one, 2 - fake only players, 3 - fake only bots, 4 - fake all
						// (obey ini file and faking command for all modes except 0)
	pingfaker_ping 80	// Base ping you want displayed by default (min: 0, max: 4095)
	pingfaker_flux 10	// Fake ping fluctuation amount by default (0 = none)
	pingfaker_self 1	// Whether to display fake ping to player himself (0 = do not)
	pingfaker_flags ""	// Affect players with these flags only (empty = all)

	
	------------
	- Commands -
	------------
	* amx_fakeping <target> <ping> [flux]
	   - Toggle fake ping on player (put pung -1 to disable faking)
	
	You can also have players automatically get fake pings by editing the
	"fakepings.ini" file in your configs folder.

	
	--------
	- ToDo -
	--------
	* Find out exactly what the arguments for the SVC_PINGS message mean, so
	   as to send a single message with all pings on it and reduce bandwidth
	   usage (does the HLSDK say anything about those?)

	
	-------------------
	- Developer Notes -
	-------------------
	The SVC_PINGS message can't be intercepted by Metamod/AMXX (it is purely
	handled by the engine) so the only way to supercede it is to send our own
	custom message right after the original is fired. This works as long as
	the custom message is parsed AFTER the original. To achieve this here, we
	send it as an unreliable message (cl_messages 1 helps see arrival order).
	
	The next difficulty is in figuring out what the message arguments are.
	For this I did some trial and error until I finally got it working, though
	in a really odd way. I also can't seem to send all of the pings in a single
	message without getting weird results or triggering heaps of message
	parsing errors (namely svc_bad).
	
	A final consideration is bandwidth usage. I found out (with cl_shownet 1)
	the packet size increases by 102 bytes when the original SVC_PINGS message
	is sent for 32 players. Sending our own message right after means the size
	will grow even larger, so we should only send the message when absolutely
	needed. In this case that's once every client data update (any less often
	than that and the ping wasn't properly overridden sometimes).

	
	-------------
	- Changelog -
	-------------
	
	* v1.0: (Feb 23, 2009)
		- Public release
	
	* v1.1: (Feb 23, 2009)
		- Managed to send up to 3 pings on a single message,
		  thus reducing bandwidth usage by 26%
	
	* v1.2: (Feb 23, 2009)
		- Added fake ping fluctuation and affect bots settings
	
	* v1.2a: (Feb 24, 2009)
		- Fixed is_user_bot flag not being reset on disconnect
	
	* v1.3: (Feb 24, 2009)
		- Added admin command to manually toggle fake ping for players
		- Added feature to automatically load fake pings from file
	
	* v1.4: (Mar 15, 2009)
		- Added feature (+CVAR) to have the fake ping be a multiple
		  of the player's real ping

	* v1.5: (Feb 13, 2010)
		- Code refactored
		- Added pingfake_mode CVAR
		- pingfake_traget CVAR renamed to pingfake_self
		- Removed unneeded CVARs
		- Ini file format: added flux
		- Made it works with any mod
		- amx_fakeping can accept flux (if not specified it will be taken from pingfake_flux)
		- Removed multiple of the player's real ping feature

	* v1.6: (Feb 23, 2010)
		- Removed admin activity show (cos this plugins needed for conspiracy and nothing more)
		- Fixed typo
	  	
=================================================================================*/

#pragma semicolon 1
#pragma ctrlchar '\'

#include <amxmodx>
#include <amxmisc>
#include <fakemeta>

#define AUTHOR "MeRcyLeZZ, Lev"
#define PLUGIN "Ping Faker"
#define VERSION "1.6"
#define VERSION_CVAR "pingfaker_version"

#define MAX_PLAYERS	32

new const FAKEPINGS_INIFILE[] = "fakepings.ini";
new const TASK_ARGUMENTS = 100;

// Plugin data
new _maxPlayers;

// Players data
new _ping[MAX_PLAYERS + 1] = { -1, ... };
new _flux[MAX_PLAYERS + 1];
new _offset[MAX_PLAYERS + 1][2];
new _argping[MAX_PLAYERS + 1][3];

// CVARs
new pcvar_mode;
new pcvar_ping;
new pcvar_flux;
new pcvar_self;
new pcvar_flags;

// Ini data
new Array:_iniAuthids;
new Array:_iniPings;
new Array:_iniFluxs;
new _iniCount;

public plugin_init()
{
	register_plugin(PLUGIN, VERSION, AUTHOR);
	register_cvar(VERSION_CVAR, VERSION, FCVAR_SPONLY | FCVAR_SERVER | FCVAR_UNLOGGED);
	
	register_concmd("amx_fakeping", "CmdFakePing", ADMIN_KICK, "<target> <ping> [flux] - Toggle fake ping on player (put ping -1 to disable faking)");
	
	pcvar_mode = register_cvar("pingfaker_mode", "1");
	pcvar_ping = register_cvar("pingfaker_ping", "80");
	pcvar_flux = register_cvar("pingfaker_flux", "10");
	pcvar_self = register_cvar("pingfaker_self", "1");
	pcvar_flags = register_cvar("pingfaker_flags", "");
	
	register_forward(FM_UpdateClientData, "fw_UpdateClientData");
	
	_maxPlayers = get_maxplayers();
	
	readIniFile();

	set_task(2.0, "CalculatePlayersPings", TASK_ARGUMENTS, _, _, "b");
}



public client_putinserver(id)
{
	CheckPlayer(id);
}

public client_authorized(id)
{
	CheckPlayer(id);
}

public client_infochanged(id)
{
	CheckPlayer(id);
}

public client_disconnect(id)
{
	_ping[id] = -1;
}

CheckPlayer(id)
{
	new mode = get_pcvar_num(pcvar_mode);
	if (mode < 1)
		return PLUGIN_CONTINUE;

	// Seek for player in config
	if (_iniCount > 0)
	{
		// Get steamId and ip
		new authid[33], ip[16], buffer[33];
		get_user_authid(id, authid, charsmax(authid));
		get_user_ip(id, ip, charsmax(ip), 1);

		for (new i = 0; i < _iniCount; i++)
		{
			// Retrieve authid
			ArrayGetString(_iniAuthids, i, buffer, charsmax(buffer));

			// Compare it with this player's steamid and ip
			if (equali(buffer, authid) || equal(buffer, ip))
			{
				_ping[id] = ArrayGetCell(_iniPings, i);
				_flux[id] = ArrayGetCell(_iniFluxs, i);
				CalculatePingArguments(id);
				return PLUGIN_CONTINUE;
			}
		}
	}

    // Get configured flags
	new flagsStr[33];
	get_pcvar_string(pcvar_flags, flagsStr, charsmax(flagsStr));
	new flags = read_flags_fixed(flagsStr);

    // Is user bot?
	new isBot = is_user_bot(id);

	// 1 - fake no one, 2 - fake only players, 3 - fake only bots, 4 - fake all
	if ((get_user_flags(id) & flags) ||
		(isBot && (mode >= 3)) ||
		(!isBot && (mode == 2 || mode == 4)))
	{
		_ping[id] = clamp(get_pcvar_num(pcvar_ping), 0, 4095);
		_flux[id] = clamp(get_pcvar_num(pcvar_flux), 0, 4095);
		CalculatePingArguments(id);
	}

	return PLUGIN_CONTINUE;
}



//****************************************
//*                                      *
//*  Fake Ping command                   *
//*                                      *
//****************************************

public CmdFakePing(id, level, cid)
{
	// Check for access flag
	if (!cmd_access(id, level, cid, 3))
		return PLUGIN_HANDLED;
	
	// Retrieve arguments
	new arg[33], player, ping, flux;
	read_argv(1, arg, charsmax(arg));
	player = cmd_target(id, arg, CMDTARGET_ALLOW_SELF);
	read_argv(2, arg, charsmax(arg));
	ping = str_to_num(arg);
	if (read_argc() > 3)
	{
		read_argv(3, arg, charsmax(arg));
		flux = str_to_num(arg);
	}
	else
		flux = get_pcvar_num(pcvar_flux);
	
	// Invalid target
	if (!player)
		return PLUGIN_HANDLED;
	
	// Get player's name for displaying/logging activity
	static name1[33], name2[33];
	get_user_name(id, name1, charsmax(name1));
	get_user_name(player, name2, charsmax(name2));
	
	// Negative value means disable fakeping
	if (ping < 0)
	{
		// Update ping overrides for player
		_ping[player] = ping;
	
		// Log activity
		static logdata[100], authid[33], ip[16];
		get_user_authid(id, authid, charsmax(authid));
		get_user_ip(id, ip, charsmax(ip), 1);
		formatex(logdata, charsmax(logdata), "ADMIN %s <%s><%s> - fake ping disabled on %s", name1, authid, ip, name2);
		log_amx(logdata);
	}
	else
	{
		// Update ping overrides for player
		_ping[player] = clamp(ping, 0, 4095);
		_flux[player] = clamp(flux, 0, 4095);
		CalculatePingArguments(player);
	
		// Log activity
		static logdata[100], authid[33], ip[16];
		get_user_authid(id, authid, charsmax(authid));
		get_user_ip(id, ip, charsmax(ip), 1);
		formatex(logdata, charsmax(logdata), "ADMIN %s <%s><%s> - fake ping of %d enabled on %s", name1, authid, ip, ping, name2);
		log_amx(logdata);
	}
	
	return PLUGIN_HANDLED;
}



//****************************************
//*                                      *
//*  Ping Faking                         *
//*                                      *
//****************************************

public CalculatePlayersPings()
{
	if (get_pcvar_num(pcvar_mode) < 1)
		return PLUGIN_CONTINUE;

	static id;
	for (id = 1; id <= _maxPlayers; id++)
	{
		if (_ping[id] < 0)
			continue;
		CalculatePingArguments(id);
	}
	return PLUGIN_CONTINUE;
}

CalculatePingArguments(id)
{
	// Calculate player's fake ping (clamp if out of bounds)
	static ping;
	ping = clamp(_ping[id] + random_num(-_flux[id], _flux[id]), 0, 4095);
	
	// First argument's ping
	for (_offset[id][0] = 0; _offset[id][0] < 4; _offset[id][0]++)
	{
		if ((ping - _offset[id][0]) % 4 == 0)
		{
			_argping[id][0] = (ping - _offset[id][0]) / 4;
			break;
		}
	}
	// Second argument's ping
	for (_offset[id][1] = 0; _offset[id][1] < 2; _offset[id][1]++)
	{
		if ((ping - _offset[id][1]) % 2 == 0)
		{
			_argping[id][1] = (ping - _offset[id][1]) / 2;
			break;
		}
	}
	// Third argument's ping
	_argping[id][2] = ping;
}

public fw_UpdateClientData(id)
{
	if (get_pcvar_num(pcvar_mode) < 1)
		return PLUGIN_CONTINUE;

	// Scoreboard key being pressed?
	if (!(pev(id, pev_button) & IN_SCORE) && !(pev(id, pev_oldbuttons) & IN_SCORE))
		return PLUGIN_CONTINUE;

	// Send fake player's pings
	static player, sending, self;
	sending = 0;
	self = get_pcvar_num(pcvar_self);
	for (player = 1; player <= _maxPlayers; player++)
	{
		// Player's ping should be real?
		if (_ping[player] < 0)
			continue;

		// Show fake ping to player himself?
		if (!self && id == player)
			continue;

		// Send message with the weird arguments
		switch (sending)
		{
			case 0:
			{
				// Start a new message
				message_begin(MSG_ONE_UNRELIABLE, SVC_PINGS, _, id);
				write_byte((_offset[player][0] * 64) + (1 + 2 * (player - 1)));
				write_short(_argping[player][0]);
				sending++;
			}
			case 1:
			{
				// Append additional data
				write_byte((_offset[player][1] * 128) + (2 + 4 * (player - 1)));
				write_short(_argping[player][1]);
				sending++;
			}
			case 2:
			{
				// Append additional data and end message
				write_byte((4 + 8 * (player - 1)));
				write_short(_argping[player][2]);
				write_byte(0);
				message_end();
				sending = 0;
			}
		}
	}

	// End message if not yet sent
	if (sending)
	{
		write_byte(0);
		message_end();
	}

	return PLUGIN_CONTINUE;
}



//****************************************
//*                                      *
//*  Load ini file                       *
//*                                      *
//****************************************

bool:readIniFile()
{
	new line, readedLength;
	new path[255];
	new buffer[255];
	new authid[33], pingStr[8], fluxStr[8];
	new ping, flux;

    // Create arrays
	_iniAuthids = ArrayCreate(33, 1);
	_iniPings = ArrayCreate(1, 1);
	_iniFluxs = ArrayCreate(1, 1);
	
	// Build file path
	get_configsdir(path, charsmax(path));
	format(path, charsmax(path), "%s/%s", path, FAKEPINGS_INIFILE);
	// Return if no ini file found
	if (!file_exists(path)) return false;

	// Parse config file
	line = 0;
	while (read_file(path, line, buffer, charsmax(buffer), readedLength))
	{
		line++;
		// Blank line or comment
		if (readedLength < 3 ||
			buffer[0] == ';' ||
			(buffer[0] == '/' && buffer[1] == '/'))
			continue;

		// Parse line
		parse(buffer, authid, charsmax(authid), pingStr, charsmax(pingStr), fluxStr, charsmax(fluxStr));
		if (!(is_str_num(pingStr))) continue;
		ping = clamp(str_to_num(pingStr), 0, 4095);
		if (!(is_str_num(fluxStr))) continue;
		flux = clamp(str_to_num(fluxStr), 0, 4095);

        // Store data into arrays
		ArrayPushString(_iniAuthids, authid);
		ArrayPushCell(_iniPings, ping);
		ArrayPushCell(_iniFluxs, flux);
		_iniCount++;
	}
	return true;
}



//****************************************
//*                                      *
//*  Other functions                     *
//*                                      *
//****************************************

/// Converts string of flags into int.
/// Only chars from 'a' to 'z' are converted, others are skipped.
stock read_flags_fixed(const flags[])
{
	new result, i = 0;
	while (flags[i] != 0)
	{
		if (flags[i] >= 'a' && flags[i] <= 'z')
			result |= 1 << (flags[i] - 'a');
		i++;	
	}
	return result;
}
